function dosyplotpub(dosydata,clev,thresh)
%   dosyplot(dosydata,clev,thresh);
%   dosyplot.m: plots dosy data as obtained from dosy_mn.m
%
%   --------------------------------INPUT------------------------------------
%   dosydata  =  The data structure from dosy_mn
%   clev      =   Number of contour levels
%   thresh    =   Threshold for contour levels as per cent of max peak height
%                 a negative value will set the default value (0.1 1% of
%                 max peak aplitude)
%
%
%   Example:
%
%   See also: dosy_mn, score_mn, decra_mn, mcr_mn, varianimport,
%             brukerimport, jeolimport, peakpick_mn, dosyplot_mn,
%             dosyresidual, dosyplot_gui, scoreplot_mn, decraplot_mn,
%             mcrplot_mn
%
%   This is a part of the GNAT
%   Copyright  2017  <Mathias Nilsson>%
%   This program is free software; you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation; either version 2 of the License, or
%   (at your option) any later version.
%
%   This program is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
%
%   You should have received a copy of the GNU General Public License along
%   with this program; if not, write to the Free Software Foundation, Inc.,
%   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
%
%   Dr. Mathias Nilsson
%   School of Chemistry, University of Manchester,
%   Oxford Road, Manchester M13 9PL, UK
%   Telephone: +44 (0) 161 306 4465
%   Fax: +44 (0)161 275 4598
% MJS     Edited line 154 'xlabel' to produce delta 1H (1 superscript)
if nargin==0
    disp(' ')
    disp(' ')
    disp(' DOSYPLOT')
    disp('At least one input is needed')
    disp(' ')
    disp(' Type <<help dosyplot>> for more info')
    disp('  ')
    return
end
if nargin >6
    error(' Too many inputs')
end
if nargin>1
    if length(clev)>1
        clvl=clev;
    else
        if clev<=0
            clev=5;
        end
    end
else
    clev=5;
end
if nargin > 2
    if thresh<0
        thresh=0.1;
    end
else
    if isfield(dosydata,'threshold')
        thresh=dosydata.threshold;
    else
        thresh=0.1;
    end
end
if length(clev)==1
    clvl=2.^(0:clev);
    clvl=clvl(1:clev);
    clvl=1./clvl;
    maxc=max(max(dosydata.DOSY));
    clvl=clvl.*maxc.*(thresh/100);
end
disp('Plotting 2D-DOSY');

hMainFig=figure('Color',[1 1 1],...
    'NumberTitle','Off',...
    'PaperOrientation','Landscape',...
    'PaperType','A4',...
    'MenuBar','none',...
    'Toolbar','Figure',...
    'Units','centimeters',...
    'Position',[0 0 29.7 21],...
    'Name','DOSY publication plot');
if isfield(dosydata,'type')
    if strcmp(dosydata.type,'lrank')
        set(hMainFig, 'Name','LRANK-DOSY')        
    end

    
%Clean up the Toolbar, removing things we don't need
tmp = findall(hMainFig,'ToolTipString','New Figure');
set(tmp,'Visible','Off')
tmp = findall(hMainFig,'ToolTipString','Open File');
set(tmp,'Visible','Off')
tmp = findall(hMainFig,'ToolTipString','Link Plot');
set(tmp,'Visible','Off')
tmp = findall(hMainFig,'ToolTipString','Insert Legend');
set(tmp,'Visible','Off')
tmp = findall(hMainFig,'ToolTipString','Rotate 3D');
set(tmp,'Visible','Off')
tmp = findall(hMainFig,'ToolTipString','Brush/Select Data');
set(tmp,'Visible','Off')
tmp = findall(hMainFig,'ToolTipString','Hide Plot Tools');
set(tmp,'Visible','Off')
tmp = findall(hMainFig,'ToolTipString','Data Cursor');
set(tmp,'Visible','Off')
tmp = findall(hMainFig,'ToolTipString','Insert Colorbar');
set(tmp,'Visible','Off')
tmp = findall(hMainFig,'ToolTipString','Edit Plot');
set(tmp,'Visible','Off')



% Two axes are needed to trick remove tickmarks top and right)
ax(1)=axes('Box','On','XTick',[],'YTick',[],'Color','None','Position',[0.1 0.1 0.75 0.70]);
ax(2)=axes('Position',[0.1 0.1 0.75 0.70]);
clvl=[clvl -clvl];

[xax,yax]=meshgrid(dosydata.Ppmscale,dosydata.Dscale);
[~,hCont]=contour(ax(2),xax,yax,(dosydata.DOSY.*(dosydata.DOSY>0))',clvl,'k','Parent',ax(2));
if dosydata.DOSY<0
    hold(ax(2),'on')
    [~,hCont]=contour(ax(2),xax,yax,(dosydata.DOSY.*(dosydata.DOSY<0))',clvl,'color',[0.75 0.75 0.75],'Parent',ax(2));
    hold(ax(2),'off')
end
set(ax(2),'TickDir','Out');

%if the data contains info on x and y limits - applky these.
if isfield(dosydata,'xlim')
    set(ax(2),'Xlim',dosydata.xlim);
end
if isfield(dosydata,'ylim')
    set(ax(2),'Ylim',dosydata.ylim);
end

set(ax(2),'LineWidth',1)
set(hCont,'LineWidth',1)
set(ax(2),'Ydir','reverse');

ylabel(ax(2),'{\fontsize{16}D  /10^{\fontsize{11}^\_10}m^{\fontsize{11}2} s^{\fontsize{11}-1} }');

set(ax(2),'Xdir','reverse');
xlabel(ax(2),'{\fontsize{16}\delta(^{1}H)}');

ax(3)=axes('Position',[0.1 0.81 0.75 0.15]);
h=plot( dosydata.Ppmscale, dosydata.Spectrum,'-k','Parent',ax(3) );


axis('tight')
axis('off')
set(h,'LineWidth',1)
set(h,'Color','black')
if isfield(dosydata,'type')
    if strcmp(dosydata.type,'lrank')
        title('\bf 2D LRANK-DOSY')
    else
        
    end
else
    
end

set(ax(2),'Box','Off', 'Color', 'None');
set(ax(2),'Xdir','reverse');
set(ax(3),'Xdir','reverse');
set(ax,'FontName','Arial','FontWeight','Normal','LineWidth',1, 'FontSize',12);
xlim(dosydata.xlim);
ylim=(dosydata.ylim);

linkaxes([ax(2) ax(3)], 'x');
end