function dosyplot_gui(dosydata,clev,thresh)

%% dosyplot_gui info(dosydata,clev,thresh);
%   dosyplot_gui.m: plots dosy data as obtained from dosy_mn.m, in an
%   interactive graphical user interface
%
%   --------------------------------INPUT------------------------------------
%   dosydata  =  The data structure from dosy_mn
%   clev      =   Number of contour levels
%   thresh    =   Threshold for contour levels as per cent of max peak height
%                 a negative value will set the default value (0.1 1% of
%                 max peak aplitude)
%
%
%   Example:
%
%   See also: dosy_mn, score_mn, decra_mn, mcr_mn, varianimport,
%             brukerimport, jeolimport, peakpick_mn, dosyplot_mn,
%             dosyresidual, dosyplot_gui, scoreplot_mn, decraplot_mn,
%             mcrplot_mn
%
%   This is a part of the GNAT
%   Copyright  2017  <Mathias Nilsson>%
%   This program is free software; you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation; either version 2 of the License, or
%   (at your option) any later version.
%
%   This program is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
%
%   You should have received a copy of the GNU General Public License along
%   with this program; if not, write to the Free Software Foundation, Inc.,
%   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
%
%   Dr. Mathias Nilsson
%   School of Chemistry, University of Manchester,
%   Oxford Road, Manchester M13 9PL, UK
%   Telephone: +44 (0) 161 306 4465
%   Fax: +44 (0)161 275 4598
%% Changes

%a=dosyguidata.immap

%use image or contour
dosyguidata.plotopt=1; %contour
%dosyguidata.plotopt=0; %imagesc
%% Initialisation;
if nargin==0
    disp(' ')
    disp(' ')
    disp(' DOSYPLOT')
    disp('At least one input is needed')
    disp(' ')
    disp(' Type <<help dosyplot>> for more info')
    disp('  ')
    return
end
if nargin >6
    error(' Too many inputs')
end

if nargin>1
    if clev<=0
        clev=7;
    end
else
    clev=7;
end
clvl=2.^(0:clev);
clvl=clvl(1:clev);
clvl=[clvl -clvl];
if nargin > 2
    if thresh<0
        thresh=0.1;
    end
else
    if (dosydata.threshold)        
        if strcmp(dosydata.type,'locodosy')
            thresh=dosydata.threshold/4;
        else
            thresh=dosydata.threshold;
        end
    else
        thresh=0.1;
    end
end

if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
        || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
    %DOSY type data
    dosy.org=dosydata.DOSY;
    
elseif strcmp(dosydata.type,'t1/t2')
    %relaxation data
    dosy.org=dosydata.T1;
else
    error('Unknown data type')
end

ppmscale.org=dosydata.Ppmscale;
spectrum.org=dosydata.Spectrum;

ScreenSize=get(0,'ScreenSize');
MaxDig=length(ppmscale.org);

%Auto set the digitisatio of the contour levels.
AutDigFac=round(MaxDig/(1.0*ScreenSize(3)));
if AutDigFac<1
    AutDigFac=1;
end
CurrDigFac=AutDigFac;
[dosydata.dosyrows,dosydata.dosycols]=size(dosy.org);

if strcmp(dosydata.type,'t1/t2')
    %relaxation data
    [dosydata.T1] = bilinearInterpolation(dosy.org, [round(dosydata.dosyrows/CurrDigFac) dosydata.dosycols]);
else
    [dosydata.DOSY] = bilinearInterpolation(dosy.org, [round(dosydata.dosyrows/CurrDigFac) dosydata.dosycols]);
end

dosydata.Spectrum=fft(ifft(spectrum.org),round(dosydata.dosyrows/CurrDigFac));
dosydata.Ppmscale=ppmscale.org(1:CurrDigFac:end);
dosydata.Ppmscale=dosydata.Ppmscale(1:round(dosydata.dosyrows/CurrDigFac));
CurrDig=length(dosydata.Ppmscale);
dosyguidata.xlim=[min(dosydata.Ppmscale) max(dosydata.Ppmscale)];
if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
        || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
    %DOSY type data
    dosyguidata.ylim=[min(dosydata.Dscale) max(dosydata.Dscale)];
    %dosyguidata.ylim=[dosydata.dmin dosydata.dmax];
elseif strcmp(dosydata.type,'t1/t2')
%relaxation data
    dosyguidata.ylim=[min(dosydata.T1scale) max(dosydata.T1scale)];
else
    error('Unknown data type')
end
 dosyguidata.imlvl=5;
 ColorLevel=2;
 BaseMult=3;

dosyguidata.immap=repmat([0,0,0],dosyguidata.imlvl,1);
for m=1:ColorLevel*BaseMult^1
    dosyguidata.immap=[dosyguidata.immap ; 0,0,1];
end
for m=1:ColorLevel*BaseMult^2
    dosyguidata.immap=[dosyguidata.immap ; 0,1,1];
end
for m=1:ColorLevel*BaseMult^3
    dosyguidata.immap=[dosyguidata.immap ; 0,1,0];
end
for m=1:ColorLevel*BaseMult^4
    dosyguidata.immap=[dosyguidata.immap ; 1,1,0];
end
for m=1:ColorLevel*BaseMult^5
    dosyguidata.immap=[dosyguidata.immap ; 1,0,0];
end
[a, ~]=size(dosyguidata.immap);
dosyguidata.imlvlTot=a;
%% Main figure

% Main figure
hMainFigure = figure(...
    'Units','pixels',...
    'MenuBar','none',...
    'Name',[upper(dosydata.type) ' plotting'],...
    'NumberTitle','Off',...
    'Toolbar','Figure',...
    'OuterPosition',[0.0 0.0 1280 1024 ],...
    'Visible','off');
set(hMainFigure,'Units','Normalized');
set(hMainFigure, 'Position',[0.1 0.4 0.6 0.7]);
hResid = axes('Units','normalized',...
    'Visible','Off',...
    'Position',[0.07 0.34 0.9 0.5]);
hDOSY = axes('Units','normalized',...
    'Position',[0.07 0.34 0.9 0.5]);
hSpec = axes('Units','normalized',...
    'Position',[0.07 0.85 0.9 0.1]);

%Clean up the Toolbar, removing things we don't need
tmp = findall(hMainFigure,'ToolTipString','New Figure');
set(tmp,'Visible','Off')
tmp = findall(hMainFigure,'ToolTipString','Open File');
set(tmp,'Visible','Off')
tmp = findall(hMainFigure,'ToolTipString','Link Plot');
set(tmp,'Visible','Off')
tmp = findall(hMainFigure,'ToolTipString','Insert Legend');
set(tmp,'Visible','Off')
% tmp = findall(hMainFigure,'ToolTipString','Edit Plot');
% set(tmp,'Visible','Off')
tmp = findall(hMainFigure,'ToolTipString','Rotate 3D');
set(tmp,'Visible','Off')
tmp = findall(hMainFigure,'ToolTipString','Brush/Select Data');
set(tmp,'Visible','Off')
tmp = findall(hMainFigure,'ToolTipString','Hide Plot Tools');
set(tmp,'Visible','Off')
tmp = findall(hMainFigure,'ToolTipString','Insert Colorbar');
set(tmp,'Visible','Off')

%% -----Panel for Plot control-------------------------------------------

% Zoom
hPlotPanel=uipanel(...
    'Parent',hMainFigure,...
    'Title','Plot control',...
    'FontWeight','bold',...
    'ForegroundColor','Blue',...
    'TitlePosition','centertop',...
    'Units','Normalized',...
    'Position',[0.01,0.01,0.4,0.25]);
hTextZoom=uicontrol(...
    'Parent',hPlotPanel,...
    'Style','text',...
    'Units','Normalized',...
    'FontWeight','bold',...
    'String','Zoom',...
    'Position',[0.15 0.9 0.2 0.1 ],...
    'horizontalalignment','c');
hZoomXYButton = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String','XY',...
    'TooltipString','Interactive zoom in the x- and y-direction',...
    'Units','normalized',...
    'Position',[0.05 0.8 0.1 0.1],...
    'Callback',@ZoomXYButton_Callback);
hZoomXButton = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String','X',...
    'TooltipString','Interactive zoom in the x-direction',...
    'Units','normalized',...
    'Position',[0.15 0.8 0.1 0.1],...
    'Callback',@ZoomXButton_Callback);
hZoomYButton = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String','Y',...
    'TooltipString','Interactive zoom in the y-direction',...
    'Units','normalized',...
    'Position',[0.25 0.8 0.1 0.1],...
    'Callback',@ZoomYButton_Callback);
hZoomOffButton = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String','off',...
    'TooltipString','Turn off the interactive zoom',...
    'Units','normalized',...
    'Position',[0.35 0.8 0.1 0.1],...
    'Callback',@ZoomOffButton_Callback);

% Pan
hTextpan=uicontrol(...
    'Parent',hPlotPanel,...
    'Style','text',...
    'Units','Normalized',...
    'FontWeight','bold',...
    'String','Pan',...
    'Position',[0.65 0.9 0.2 0.1 ],...
    'horizontalalignment','c');
hpanXYButton = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String','XY',...
    'TooltipString','Interactive pan in the x- and y-direction',...
    'Units','normalized',...
    'Position',[0.55 0.8 0.1 0.1],...
    'Callback',@panXYButton_Callback);
hpanXButton = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String','X',...
    'TooltipString','Interactive pan in the x-direction',...
    'Units','normalized',...
    'Position',[0.65 0.8 0.1 0.1],...
    'Callback',@panXButton_Callback);
hpanYButton = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String','Y',...
    'TooltipString','Interactive pan in the y-direction',...
    'Units','normalized',...
    'Position',[0.75 0.8 0.1 0.1],...
    'Callback',@panYButton_Callback);
hpanOffButton = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String','off',...
    'TooltipString','Turn off the interactive pan',...
    'Units','normalized',...
    'Position',[0.85 0.8 0.1 0.1],...
    'Callback',@panOffButton_Callback);

% Scale
hTextScale=uicontrol(...
    'Parent',hPlotPanel,...
    'Style','text',...
    'Units','Normalized',...
    'FontWeight','bold',...
    'String','Scale',...
    'Position',[0.2 0.65 0.2 0.1 ],...
    'horizontalalignment','l');
hButtonMult2 = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String',[char(hex2dec('D7')) ' 2'],...
    'Units','Normalized',...
    'TooltipString','Increase the vertical scale by a factor 2',...
    'Position',[0.05 0.5 0.1 0.15],...
    'Callback',{@ButtonMult2_Callback});
hButtonDiv2 = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String',[char(hex2dec('F7')) ' 2'],...
    'Units','Normalized',...
    'TooltipString','Decrease the vertical scale by a factor 2',...
    'Position',[0.05 0.35 0.1 0.15],...
    'Callback',{@ButtonDiv2_Callback});
hButtonMult11 = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String',[char(hex2dec('D7')) ' 1.1'],...
    'Units','Normalized',...
    'TooltipString','Increase the vertical scale by a factor 1.1',...
    'Position',[0.15 0.5 0.1 0.15],...
    'Callback',{@ButtonMult11_Callback});
hButtonDiv11 = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','PushButton',...
    'String',[char(hex2dec('F7')) ' 1.1'],...
    'Units','Normalized',...
    'TooltipString','Decrease the vertical scale by a factor 1.1',...
    'Position',[0.15 0.35 0.1 0.15],...
    'Callback',{@ButtonDiv11_Callback});
hlogscaleGroup=uibuttongroup(...
    'Visible','On',...
    'Parent',hPlotPanel,...
    'Title','',...
    'SelectionChangeFcn', {@Scale_Callback},...
    'TitlePosition','centertop',...
    'Units','Normalized',...
    'Position',[0.27 0.35 0.16 0.3]);
hRadioLog = uicontrol(...
    'Parent',hlogscaleGroup,...
    'Style','RadioButton',...
    'String','Logarithmic',...
    'Units','normalized',...
    'Position',[0.05 0.65 0.9 0.3]);
hRadioLin = uicontrol(...
    'Parent',hlogscaleGroup,...
    'Style','RadioButton',...
    'String','Linear',...
    'Value',1,...
    'Units','normalized',...
    'Position',[0.05 0.15 0.9 0.3]);

% Plot
hButtonOrgscale = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','Pushbutton',...
    'String','Original',...
    'Units','Normalized',...
    'TooltipString','Plot original data',...
    'Position',[0.05 0.19 0.2 0.15],...
    'Callback',{@ButtonOrgscale_Callback});
hButtonReplot = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','Pushbutton',...
    'String','Replot',...
    'TooltipString','Plot processed data',...
    'Units','Normalized',...
    'Position',[0.25 0.19 0.2 0.15],...
    'Callback',{@ButtonReplot_Callback});
hButtonStats = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','Pushbutton',...
    'String','Stats',...
    'Units','Normalized',...
    'TooltipString','Save fitting results in a .txt file',...
    'Position',[0.1 0.02 0.25 0.15],...
    'Callback',{@ButtonStats_Callback});

hButtonPlotSep = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','Pushbutton',...
    'String','Separate plot',...
    'TooltipString','Plot spectrum in a separate window',...
    'Units','Normalized',...
    'Position',[0.5 0.5 0.25 0.15],...
    'Callback',{@ButtonPlotSep_Callback});
hButtonPlotPub = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','Pushbutton',...
    'String','Publication plot',...
    'TooltipString','Plot spectrum for publication',...
    'Units','Normalized',...
    'Position',[0.75 0.5 0.25 0.15],...
    'Callback',{@ButtonPlotPub_Callback});


% Projections
uicontrol(...
    'Parent',hPlotPanel,...
    'Style','text',...
    'Units','Normalized',...
    'String','Projections',...
    'FontWeight','bold',...
    'Position',[0.64 0.35 0.2 0.1 ],...
    'horizontalalignment','c');
hProjGroup=uibuttongroup(...
    'Visible','On',...
    'Parent',hPlotPanel,...
    'Title','',...
    'TitlePosition','centertop',...
    'BorderType','None',...
    'Units','Normalized',...
    'SelectionChangeFcn', {@Projection_Callback},...
    'Position',[0.6 0.2 0.325 0.15]);
hRadioSky = uicontrol(...
    'Parent',hProjGroup,...
    'Style','RadioButton',...
    'String','Sky',...
    'Units','normalized',...
    'Position',[0.0 0.2 0.4 0.9]);
hRadioSum = uicontrol(...
    'Parent',hProjGroup,...
    'Style','RadioButton',...
    'String','Sum',...
    'Value',1,...
    'Units','normalized',...
    'Position',[0.35 0.2 0.4 0.9]);
hRadioNone = uicontrol(...
    'Parent',hProjGroup,...
    'Style','RadioButton',...
    'String','None',...
    'Units','normalized',...
    'Position',[0.7 0.2 0.4 0.9]);
hButtonPlotDproj = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','Pushbutton',...
    'String','Y-dir',...
    'Units','Normalized',...
    'TooltipString','Plot Y-projection',...
    'Position',[0.5 0.03 0.25 0.15],...
    'Callback',{@ButtonPlotDproj_Callback});
hButtonPlotDSpecproj = uicontrol(...
    'Parent',hPlotPanel,...
    'Style','Pushbutton',...
    'String','X-dir',...
    'Units','Normalized',...
    'TooltipString','Plot X-projection',...
    'Position',[0.75 0.03 0.25 0.15],...
    'Callback',{@ButtonPlotSpecproj_Callback});
%% -----Panel for Spectrum digitisation-------------------------------------------
hDigPanel=uipanel(...
    'Parent',hMainFigure,...
    'Title','Spectrum digitisation',...
    'FontWeight','bold',...
    'ForegroundColor','Blue',...
    'TitlePosition','centertop',...
    'Visible','On',...
    'Units','Normalized',...
    'Position',[0.42,0.01,0.16,0.25]);
hButtonDigAuto = uicontrol(...
    'Parent',hDigPanel,...
    'Style','PushButton',...
    'String','Auto',...
    'Units','normalized',...
    'TooltipString','Automatic digitisation in chemical shift dimension',...
    'Position',[0.1 0.75 0.38 0.15],...
    'Callback',{@ButtonDigAuto_Callback});
hButtonDigMax = uicontrol(...
    'Parent',hDigPanel,...
    'Style','PushButton',...
    'String','Max.',...
    'Units','normalized',...
    'TooltipString','Maximum digitisation in chemical shift dimension',...
    'Position',[0.5 0.75 0.38 0.15],...
    'Callback',{@ButtonDigMax_Callback});
hButtonDigMinus = uicontrol(...
    'Parent',hDigPanel,...
    'Style','PushButton',...
    'String',[char(hex2dec('F7')) ' 2'],...
    'Units','normalized',...
    'TooltipString','Increase digitisation in chemical shift dimension by a factor 2',...
    'Position',[0.1 0.57 0.38 0.15],...
    'Callback',{@ButtonDigMinus_Callback});
hButtonDigPlus = uicontrol(...
    'Parent',hDigPanel,...
    'Style','PushButton',...
    'String',[char(hex2dec('D7')) ' 2'],...
    'Units','normalized',...
    'TooltipString','Decrease digitisation in chemical shift dimension by a factor 2',...
    'Position',[0.5 0.57 0.38 0.15],...
    'Callback',{@ButtonDigPlus_Callback});
hEditDig1=uicontrol(...
    'Parent',hDigPanel,...
    'Style','edit',...
    'Units','Normalized',...
    'BackgroundColor','w',...
    'Enable','Off',...
    'String',MaxDig,...
    'Position',[0.1 0.28 0.8 0.15 ],...
    'CallBack', {@EditDig1_Callback}); 
hTextDig1=uicontrol(...
    'Parent',hDigPanel,...
    'Style','text',...
    'Units','Normalized',...
    'String','Max.',...
    'TooltipString','Maximum digitisation in chemical shift dimension',...
    'Position',[0.1 0.43 0.8 0.1 ],...
    'horizontalalignment','Center');
hTextDig2=uicontrol(...
    'Parent',hDigPanel,...
    'Style','text',...
    'Units','Normalized',...
    'String','Current',...
    'TooltipString','Current digitisation in chemical shift dimension',...
    'Position',[0.1 0.15 0.8 0.1 ],...
    'horizontalalignment','Center');
hEditDig2=uicontrol(...
    'Parent',hDigPanel,...
    'Style','edit',...
    'Units','Normalized',...
    'BackgroundColor','w',...
    'Enable','Off',...
    'String',CurrDig,...
    'Position',[0.1 0.01 0.8 0.15 ],...
    'horizontalalignment','Center');
%% -----Panel for Residuals-----------------------------------------
hResidPanel=uipanel(...
    'Parent',hMainFigure,...
    'Title','Residuals',...
    'FontWeight','bold',...
    'ForegroundColor','Blue',...
    'TitlePosition','centertop',...
    'Visible','Off',...
    'Units','Normalized',...
    'Position',[0.59,0.01,0.4,0.25]);

% Plot
hButtonPlotResid = uicontrol(...
    'Parent',hResidPanel,...
    'Style','PushButton',...
    'String','Plot',...
    'TooltipString','Plot residuals in current window',...
    'Units','normalized',...
    'Position',[0.12 0.75 0.22 0.20],...
    'Callback',{@ButtonPlotResid_Callback}); 
hButtonPlotResidSep = uicontrol(...
    'Parent',hResidPanel,...
    'Style','PushButton',...
    'String','Separate plot',...
    'Units','normalized',...
    'TooltipString','Plot residuals in a separate window',...
    'Position',[0.1 0.5 0.28 0.15],...
    'Callback',{@ButtonPlotResidSep_Callback});

% Peak number
hEditFlip=uicontrol(...
    'Parent',hResidPanel,...
    'Style','edit',...
    'Units','Normalized',...
    'BackgroundColor','w',...
    'String',1,...
    'Position',[0.18 0.1 0.1 0.1 ],...
    'CallBack', {@EditFlip_Callback});
hButtonFlipPlus = uicontrol(...
    'Parent',hResidPanel,...
    'Style','PushButton',...
    'String','+1',...
    'Units','normalized',...
    'Position',[0.3 0.1 0.06 0.1],...
    'Callback',{@ButtonFlipPlus_Callback});                            
hButtonFlipMinus = uicontrol(...
    'Parent',hResidPanel,...
    'Style','PushButton',...
    'String','-1',...
    'Units','normalized',...
    'Position',[0.1 0.1 0.06 0.1],...
    'Callback',{@ButtonFlipMinus_Callback});                           
hTextPeakNumber=uicontrol(...
    'Parent',hResidPanel,...
    'Style','text',...
    'Units','Normalized',...
    'String','Peak number',...
    'FontWeight','bold',...
    'Position',[0.03 0.25 0.4 0.1 ],...
    'horizontalalignment','c');

% Compare fits
hButtonCompareResid = uicontrol(...
    'Parent',hResidPanel,...
    'Style','PushButton',...
    'String','Compare fits',...
    'Units','normalized',...
    'TooltipString','Plot fit comparison',...
    'Position',[0.6 0.8 0.28 0.15],...
    'Callback',{@ButtonCompareResid_Callback});
hEditComp1=uicontrol(...
    'Parent',hResidPanel,...
    'Style','edit',...
    'Units','Normalized',...
    'BackgroundColor','w',...
    'Enable','Off',...
    'String',1,...
    'Position',[0.6 0.62 0.1 0.1 ],...
    'CallBack', {@EditComp1_Callback});
hEditComp2=uicontrol(...
    'Parent',hResidPanel,...
    'Style','edit',...
    'Units','Normalized',...
    'BackgroundColor','w',...
    'Enable','Off',...
    'String',1,...
    'Position',[0.6 0.5 0.1 0.1 ],...
    'CallBack', {@EditComp2_Callback});
hEditComp3=uicontrol(...
    'Parent',hResidPanel,...
    'Style','edit',...
    'Units','Normalized',...
    'BackgroundColor','w',...
    'Enable','Off',...
    'String',1,...
    'Position',[0.6 0.38 0.1 0.1 ],...
    'CallBack', {@EditComp3_Callback});
hEditComp4=uicontrol(...
    'Parent',hResidPanel,...
    'Style','edit',...
    'Units','Normalized',...
    'BackgroundColor','w',...
    'Enable','Off',...
    'String',1,...
    'Position',[0.6 0.26 0.1 0.1 ],...
    'CallBack', {@EditComp4_Callback});
hEditComp5=uicontrol(...
    'Parent',hResidPanel,...
    'Style','edit',...
    'Units','Normalized',...
    'BackgroundColor','w',...
    'Enable','Off',...
    'String',1,...
    'Position',[0.6 0.14 0.1 0.1 ],...
    'CallBack', {@EditComp5_Callback});
hCheckComp1 = uicontrol(...
    'Parent',hResidPanel,...
    'Style','Checkbox',...
    'String','',...
    'Units','normalized',...
    'Value',0,...
    'Position',[0.54 0.62 0.05 0.1],...
    'Callback',{@CheckComp1_Callback});
hCheckComp2 = uicontrol(...
    'Parent',hResidPanel,...
    'Style','Checkbox',...
    'String','',...
    'Units','normalized',...
    'Value',0,...
    'Position',[0.54 0.50 0.05 0.1],...
    'Callback',{@CheckComp2_Callback});
hCheckComp3 = uicontrol(...
    'Parent',hResidPanel,...
    'Style','Checkbox',...
    'String','',...
    'Units','normalized',...
    'Value',0,...
    'Position',[0.54 0.38 0.05 0.1],...
    'Callback',{@CheckComp3_Callback});
hCheckComp4 = uicontrol(...
    'Parent',hResidPanel,...
    'Style','Checkbox',...
    'String','',...
    'Units','normalized',...
    'Value',0,...
    'Position',[0.54 0.26 0.05 0.1],...
    'Callback',{@CheckComp4_Callback});
hCheckComp5 = uicontrol(...
    'Parent',hResidPanel,...
    'Style','Checkbox',...
    'String','',...
    'Units','normalized',...
    'Value',0,...
    'Position',[0.54 0.14 0.05 0.1],...
    'Callback',{@CheckComp5_Callback});
hTextData=uicontrol(...
    'Parent',hResidPanel,...
    'Style','text',...
    'Units','Normalized',...
    'String','Data type',...
    'FontWeight','bold',...
    'Position',[0.76 0.64 0.2 0.1 ],...
    'horizontalalignment','l'); %#ok<NASGU>
hCheckOrig = uicontrol(...
    'Parent',hResidPanel,...
    'Style','Checkbox',...
    'String','Original',...
    'Units','normalized',...
    'Value',0,...
    'Position',[0.75 0.50 0.2 0.1],...
    'Callback',{@CheckOrig_Callback});
hCheckFitted = uicontrol(...
    'Parent',hResidPanel,...
    'Style','Checkbox',...
    'String','Fitted',...
    'Units','normalized',...
    'Value',0,...
    'Position',[0.75 0.35 0.2 0.1],...
    'Callback',{@CheckFitted_Callback});
hCheckResid = uicontrol(...
    'Parent',hResidPanel,...
    'Style','Checkbox',...
    'String','Residual',...
    'Units','normalized',...
    'Value',0,...
    'Position',[0.75 0.20 0.2 0.1],...
    'Callback',{@CheckResid_Callback});

if strcmp(dosydata.type,'dosy') || strcmp(dosydata.type,'ILT') || strcmp(dosydata.type,'t1/t2')
    set(hResidPanel,'Visible','On');
end

if strcmp(dosydata.type,'t1/t2')
    maxc=max(max(abs(dosydata.T1)));
else
    maxc=max(max(abs(dosydata.DOSY)));
end
dosyguidata.clvl=clvl.*maxc.*(thresh/100);

dosyguidata.residpeak=1;
dosyguidata.comparepeaks=[1 1 1 1 1];
dosyguidata.compareuse=[0 0 0 0 0];
dosyguidata.comparetype=[0 0 0];

guidata(hMainFigure,dosyguidata);
xlim([min(dosydata.Ppmscale) max(dosydata.Ppmscale)])
ylim(dosyguidata.ylim)
PlotSpectrum();
PlotContour();
set(hDOSY, 'Ylim',dosyguidata.ylim,'Xlim',dosyguidata.xlim);

movegui(hMainFigure,'center')
movegui(hMainFigure,'north')
set(hMainFigure,'Visible','on')
%% -----Digitisation Callbacks-------------------------------------------
    function ButtonDigAuto_Callback(source,eventdata)%#ok
        dosyguidata=guidata(hMainFigure);
        CurrDigFac=AutDigFac;
        if strcmp(dosydata.type,'t1/t2')
            %relaxation data
            [dosydata.T1] = bilinearInterpolation(dosy.org, [round(dosydata.dosyrows/CurrDigFac) dosydata.dosycols]);
        else
            [dosydata.DOSY] = bilinearInterpolation(dosy.org, [round(dosydata.dosyrows/CurrDigFac) dosydata.dosycols]);
        end
        dosydata.Spectrum=fft(ifft(spectrum.org),round(dosydata.dosyrows/CurrDigFac));
        dosydata.Ppmscale=ppmscale.org(1:CurrDigFac:end);
        dosydata.Ppmscale=dosydata.Ppmscale(1:round(dosydata.dosyrows/CurrDigFac));
        CurrDig=length(dosydata.Ppmscale);
        
        set(hEditDig2,'String',CurrDig);
        guidata(hMainFigure,dosyguidata);
        PlotContour();
        PlotSpectrum();
    end
    function ButtonDigMax_Callback(source,eventdata)%#ok
        dosyguidata=guidata(hMainFigure);
        if strcmp(dosydata.type,'t1/t2')
            %relaxation data
            dosydata.T1=dosy.org;
        else
            dosydata.DOSY=dosy.org;
        end
        dosydata.DOSY=dosy.org;
        dosydata.Ppmscale=ppmscale.org;
        dosydata.Spectrum=spectrum.org;
        CurrDig=length(dosydata.Ppmscale);
        CurrDigFac=1;
        set(hEditDig2,'String',CurrDig);
        guidata(hMainFigure,dosyguidata);
        PlotContour();
        PlotSpectrum();
        guidata(hMainFigure,dosyguidata);
        
    end
    function ButtonDigMinus_Callback(source,eventdata)%#ok
        dosyguidata=guidata(hMainFigure);
        CurrDigFac=2*CurrDigFac;
        if strcmp(dosydata.type,'t1/t2')
            %relaxation data
            [dosydata.T1] = bilinearInterpolation(dosy.org, [round(dosydata.dosyrows/CurrDigFac) dosydata.dosycols]);
        else
            [dosydata.DOSY] = bilinearInterpolation(dosy.org, [round(dosydata.dosyrows/CurrDigFac) dosydata.dosycols]);
        end
        dosydata.Spectrum=fft(ifft(spectrum.org),round(dosydata.dosyrows/CurrDigFac));
        
        
        dosydata.Ppmscale=ppmscale.org(1:CurrDigFac:end);
        dosydata.Ppmscale=dosydata.Ppmscale(1:round(dosydata.dosyrows/CurrDigFac));
        CurrDig=length(dosydata.Ppmscale);
        set(hEditDig2,'String',CurrDig);
        guidata(hMainFigure,dosyguidata);
        PlotContour();
        PlotSpectrum();
    end
    function ButtonDigPlus_Callback(source,~)
        dosyguidata=guidata(hMainFigure);
        CurrDigFac=round(CurrDigFac/2);
        if CurrDigFac < 1
            CurrDigFac=1;
        end
        if strcmp(dosydata.type,'t1/t2')
            %relaxation data
            [dosydata.T1] = bilinearInterpolation(dosy.org, [round(dosydata.dosyrows/CurrDigFac) dosydata.dosycols]);
        else
            [dosydata.DOSY] = bilinearInterpolation(dosy.org, [round(dosydata.dosyrows/CurrDigFac) dosydata.dosycols]);
        end
        dosydata.Spectrum=fft(ifft(spectrum.org),round(dosydata.dosyrows/CurrDigFac));
        dosydata.Ppmscale=ppmscale.org(1:CurrDigFac:end);
        dosydata.Ppmscale=dosydata.Ppmscale(1:round(dosydata.dosyrows/CurrDigFac));
        CurrDig=length(dosydata.Ppmscale);
        set(hEditDig2,'String',CurrDig);
        guidata(hMainFigure,dosyguidata);
        PlotContour();
        PlotSpectrum();
    end
%% -----Plot Control Callbacks-------------------------------------------
    function ZoomXYButton_Callback(source,eventdata)%#ok
        zoom off
        pan off
        zoom on
    end
    function ZoomXButton_Callback(source,eventdata)%#ok
        zoom off
        pan off
        zoom xon
    end
    function ZoomYButton_Callback(source,eventdata)%#ok
        zoom off
        pan off
        zoom yon
    end
    function ZoomOffButton_Callback(source,eventdata)%#ok
        zoom off
        pan off
    end
    function panXYButton_Callback(source,eventdata)%#ok
        pan off
        zoom off
        pan on
    end
    function panXButton_Callback(source,eventdata)%#ok
        pan off
        zoom off
        pan xon
    end
    function panYButton_Callback(source,eventdata)%#ok
        pan off
        zoom off
        pan yon
    end
    function panOffButton_Callback(source,eventdata)%#ok
        pan off
        zoom off
    end
    function ButtonMult2_Callback(source,eventdata)%#ok
        %tic
        hContour1= findobj('Tag','DOSYContour1');
        hContour2= findobj('Tag','DOSYContour2');

        if dosyguidata.plotopt==0  %use imagesc
            for k=1:5
                dosyguidata.immap(1,:)=[];
                dosyguidata.immap=[dosyguidata.immap ; 0,0,1];
            end
            
            guidata(hMainFigure,dosyguidata);
            colormap(dosyguidata.immap)
        end
        dosyguidata=guidata(hMainFigure);
        dosyguidata.clvl=dosyguidata.clvl/2;
        set(hContour1,'LevelList',dosyguidata.clvl)
        set(hContour2,'LevelList',dosyguidata.clvl)
        guidata(hMainFigure,dosyguidata);
        %PlotContour()
        %toc
    end
    function ButtonMult11_Callback(source,eventdata)%#ok
        hContour1= findobj('Tag','DOSYContour1');
        hContour2= findobj('Tag','DOSYContour2');
        dosyguidata=guidata(hMainFigure);
        dosyguidata.clvl=dosyguidata.clvl/1.1;
        set(hContour1,'LevelList',dosyguidata.clvl)
        set(hContour2,'LevelList',dosyguidata.clvl)
        if dosyguidata.plotopt==0  %use imagesc
            %if 
            for k=1:1
                dosyguidata.immap(1,:)=[];
               % dosyguidata.immap=[dosyguidata.immap ; 0,0,1];
            end
          %  a=dosyguidata.immap
            size(dosyguidata.immap)
           guidata(hMainFigure,dosyguidata)
            colormap(dosyguidata.immap)
        end

        %PlotContour()
        guidata(hMainFigure,dosyguidata);
        
    end
    function ButtonDiv2_Callback(source,eventdata)%#ok
        %tic
        hContour1= findobj('Tag','DOSYContour1');
        hContour2= findobj('Tag','DOSYContour2');
        
        if dosyguidata.plotopt==0  %use imagesc
            for k=1:5
                dosyguidata.immap(end,:)=[];
                dosyguidata.immap=[ 1,1,1; dosyguidata.immap ];
            end
            guidata(hMainFigure,dosyguidata);
            colormap(dosyguidata.immap)
        end
        
        %dosyguidata=guidata(hMainFigure);
           guidata(hMainFigure,dosyguidata);
        dosyguidata.clvl=dosyguidata.clvl*2;
        set(hContour1,'LevelList',dosyguidata.clvl)
        set(hContour2,'LevelList',dosyguidata.clvl)
        guidata(hMainFigure,dosyguidata);
        %PlotContour()
        %toc
    end
    function ButtonDiv11_Callback(source,eventdata)%#ok
        hContour1= findobj('Tag','DOSYContour1');
        hContour2= findobj('Tag','DOSYContour2');
        dosyguidata=guidata(hMainFigure);
        dosyguidata.clvl=dosyguidata.clvl*1.1;
        set(hContour1,'LevelList',dosyguidata.clvl)
        set(hContour2,'LevelList',dosyguidata.clvl)
        if dosyguidata.plotopt==0  %use imagesc
            for k=1:1
               % dosyguidata.immap(end,:)=[];
                dosyguidata.immap=[0,0,0; dosyguidata.immap ] ;
            end
            guidata(hMainFigure,dosyguidata);
            colormap(dosyguidata.immap)
            % PlotContour()
        end
        guidata(hMainFigure,dosyguidata);
    end
    function ButtonOrgscale_Callback(source,eventdata) %#ok
        dosyguidata=guidata(hMainFigure);
        if strcmp(dosydata.type,'t1/t2')
            maxc=max(max(abs(dosydata.T1)));
        else
            maxc=max(max(abs(dosydata.DOSY)));
        end
        dosyguidata.clvl=clvl.*maxc.*(thresh/100);
        guidata(hMainFigure,dosyguidata);
        xlim([min(dosydata.Ppmscale) max(dosydata.Ppmscale)])
        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            %DOSY type data
            ylim([min(dosydata.Dscale) max(dosydata.Dscale)])
        elseif strcmp(dosydata.type,'t1/t2')
            %relaxation data
            ylim([min(dosydata.T1scale) max(dosydata.T1scale)])
        else
            error('Unknown data type')
        end
        PlotContour()
        
        PlotSpectrum()
        guidata(hMainFigure,dosyguidata);
    end
    function ButtonReplot_Callback(source,eventdata)
        dosyguidata=guidata(hMainFigure);
        xlim(dosyguidata.xlim);
        ylim(dosyguidata.ylim);
        PlotContour();
    end
    function ButtonPlotSep_Callback(source,eventdata) 
        dosyguidata=guidata(hMainFigure);
        dosydata.xlim=xlim(hDOSY);
        dosydata.ylim=ylim(hDOSY);
        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            dosyplot(dosydata,dosyguidata.clvl);
        elseif strcmp(dosydata.type,'t1/t2')
            t1plot(dosydata,dosyguidata.clvl);
        else
            error('Unknown data type')
        end
    end
    function ButtonPlotPub_Callback(source,eventdata) 
        dosyguidata=guidata(hMainFigure);
        dosydata.xlim=xlim(hDOSY);
        dosydata.ylim=ylim(hDOSY);

        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            dosyplotpub(dosydata,dosyguidata.clvl);
        elseif strcmp(dosydata.type,'t1/t2')
            %dosydata;
            t1plotpub(dosydata,dosyguidata.clvl);
        else
            error('Unknown data type')
        end
    end
    function ButtonStats_Callback(~,~)
        DTpath=which('GNAT');
        DTpath=DTpath(1:(end-6));
        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            UseFile=[DTpath 'dosystats.txt'];
            DefName='dosystats.txt';
            %type([DTpath 'dosystats.txt']);
        elseif strcmp(dosydata.type,'t1/t2')
            %type([DTpath 't1stats.txt']);
            UseFile=[DTpath 'relaxationstats.txt'];
            DefName='relaxationstats.txt';
        else
            error('Unknown data type')
        end
        type(UseFile)
        
        [filename, pathname] = uiputfile('*.txt','Save statistics file',DefName);
            if filename==0
                return
            end
        copyfile(UseFile,[pathname filename],'f')
    end
    function Scale_Callback(~,~)
        switch get(hlogscaleGroup,'SelectedObject')
            case hRadioLog
                dosyguidata=guidata(hMainFigure);
                set(hDOSY,'YScale','log');
            case hRadioLin
                dosyguidata=guidata(hMainFigure);
                set(hDOSY,'YScale','linear');
            otherwise
                error('illegal choice')
        end
        
    end
    function ButtonPlotSpecproj_Callback(source,eventdata) 
        dosyguidata=guidata(hMainFigure);
        dosydata.xlim=xlim(hDOSY);
        dosydata.ylim=ylim(hDOSY);

        temp_ylim=ylim(hDOSY);
        temp_xlim=xlim(hDOSY);
        
        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            DOSY=dosydata.DOSY;
            
            high_y=find(dosydata.Dscale>temp_ylim(2));
            low_y=find(dosydata.Dscale<temp_ylim(1));
            
        elseif strcmp(dosydata.type,'t1/t2')
            DOSY=dosydata.T1;
            
            high_y=find(dosydata.T1scale>temp_ylim(2));
            low_y=find(dosydata.T1scale<temp_ylim(1));
        else
            error('Unknown data type')
        end
        
        xprojFigure=figure('Color',[1 1 1],...
            'NumberTitle','Off',...
            'Units','centimeters',...
            'MenuBar','none',...
            'Toolbar','Figure',...
            'Position',[0 0 29.7 21],...
            'Name','Spectral projection');
        
               
            % Clean up the Toolbar, removing things we don't need
            tmp = findall(xprojFigure,'ToolTipString','New Figure');
            set(tmp,'Visible','Off')
%             tmp = findall(xprojFigure,'ToolTipString','Edit Plot');
%             set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Zoom In');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Zoom Out');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Pan');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Data Cursor');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Open File');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Link Plot');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Insert Legend');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Rotate 3D');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Brush/Select Data');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Hide Plot Tools');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Data Cursor');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Insert Colorbar');
            set(tmp,'Visible','Off')
            tmp = findall(xprojFigure,'ToolTipString','Show Plot Tools and Dock Figure');
            set(tmp,'Visible','Off')

        
        specscale=dosydata.Ppmscale;        
        DOSY(:,[high_y low_y])=[];

        high_x=find(specscale>temp_xlim(2));
        low_x=find(specscale<temp_xlim(1));
        specscale([high_x' low_x'])=[];
        DOSY([high_x' low_x'],:)=[];
        
        switch get(hProjGroup,'SelectedObject')
            case hRadioSky
                
                if max(max(DOSY))>abs(min(min(DOSY))) %
                    specproj=max(DOSY,[],2);
                else
                    specproj=-min(DOSY,[],2);
                end
                
                Ylab='Skyline';
                
            case hRadioSum
                
                if max(max(DOSY))>abs(min(min(DOSY))) %
                    specproj=sum(DOSY,2);
                else
                    specproj=-sum(DOSY,2);
                end
                Ylab='Sum';
                
            otherwise
                error('illegal choice')
        end
        
        
        plot(specscale,specproj,'LineWidth',1,'Color','black');
        set(gca,'Xdir','reverse');
        axis('tight')
        xlabel('\fontname{ariel} \bf Chemical shift /ppm');
        ylabel(['\fontname{ariel} \bf' Ylab ]);
        set(gca,'LineWidth',1);
        
        if max(specproj)>0
            ylim([0-max(specproj)*0.1  max(specproj)*1.1]);
        else
            ylim([min(specproj)*1.1  0+min(specproj)*0.1]);
        end
    end
    function ButtonPlotDproj_Callback(source,eventdata)
        dosyguidata=guidata(hMainFigure);
        temp_ylim=ylim(hDOSY);
        temp_xlim=xlim(hDOSY);
        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            projname='Diffusion projection';
            x_string='\bf Diffusion coefficient / 10^{-10} m^{2} s^{-1} ';
            DOSY=dosydata.DOSY;
            specscale=dosydata.Dscale;
            high_y=find(dosydata.Dscale>temp_ylim(2));
            low_y=find(dosydata.Dscale<temp_ylim(1));
            
        elseif strcmp(dosydata.type,'t1/t2')
            if dosydata.FitType == 3
                projname='T2 Projection';
                x_string='\bf T_2 /s';
            else
                projname='T1 Projection';
                x_string='\bf T_1 /s';
            end

            DOSY=dosydata.T1;
            specscale=dosydata.T1scale;
            high_y=find(dosydata.T1scale>temp_ylim(2));
            low_y=find(dosydata.T1scale<temp_ylim(1));
        else
            error('Unknown data type')
        end
        
        yprojFigure = figure('Color',[1 1 1],...
            'NumberTitle','Off',...
            'MenuBar','none',...
            'Units','centimeters',...
            'Toolbar','Figure',...
            'Position',[0 0 29.7 21],...
            'Name',projname);
        
        
        
% Clean up the Toolbar, removing things we don't need
tmp = findall(yprojFigure,'ToolTipString','New Figure');
set(tmp,'Visible','Off')
% tmp = findall(yprojFigure,'ToolTipString','Edit Plot');
% set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Zoom In');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Zoom Out');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Pan');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Data Cursor');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Open File');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Link Plot');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Insert Legend');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Rotate 3D');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Brush/Select Data');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Hide Plot Tools');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Data Cursor');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Insert Colorbar');
set(tmp,'Visible','Off')
tmp = findall(yprojFigure,'ToolTipString','Show Plot Tools and Dock Figure');
set(tmp,'Visible','Off')

        
        DOSY(:, [ low_y high_y])=[];
        specscale([ low_y  high_y])=[];       
        
        high_x=find(dosydata.Ppmscale>temp_xlim(2));
        low_x=find(dosydata.Ppmscale<temp_xlim(1));       
        
        DOSY([ low_x(:) ; high_x(:) ],:)=[];        
        
        switch get(hProjGroup,'SelectedObject')
            case hRadioSky
                if max(max(DOSY))>abs(min(min(DOSY)))
                    specproj=max(DOSY);
                else
                    specproj=-min(DOSY);
                end
                Ylab='Skyline';
            case hRadioSum
                if max(max(DOSY))>abs(min(min(DOSY))) %
                    %disp('hi')
                    specproj=sum(DOSY);
                else
                    %disp('hi2')
                    specproj=-sum(DOSY);
                end
                Ylab='Sum';
            otherwise
                error('illegal choice')
        end
        
        plot(specscale,specproj,'LineWidth',1,'Color','black');
        set(gca,'Xdir','reverse');
        axis('tight')
        xlabel(x_string);
        ylabel(['\fontname{ariel} \bf' Ylab ]);
        set(gca,'LineWidth',1);
        ylim([0-max(specproj)*0.1  max(specproj)*1.1]);
        
    end
%---------End of Plot Control Callbacks------------------------------------
%% -----Plot Residual Callbacks------------------------------------------
    function Projection_Callback(source,eventdata)
        dosyguidata=guidata(hMainFigure);

        switch get(hProjGroup,'SelectedObject')
            case hRadioSum
            case hRadioSky
            case hRadioNone
        end
        
        guidata(hMainFigure,dosyguidata);
    end
    function ButtonPlotResid_Callback(source,eventdata) %#ok
        dosyguidata=guidata(hMainFigure);
        PlotResidual(dosyguidata.residpeak);
    end
    function ButtonPlotResidSep_Callback(source,eventdata) %#ok
        dosyguidata=guidata(hMainFigure);
        dosyresidual(dosydata,dosyguidata);
    end
    function EditComp1_Callback(source,eventdata) %#ok<INUSD>
        dosyguidata=guidata(hMainFigure);
        
        if isnan((str2double(get(hEditComp1,'string'))))
            disp('Must be a number')
            set(hEditComp1,'String',num2str(dosyguidata.comparepeaks(1)))
        else
            
            dosyguidata.comparepeaks(1)=...
                round((str2double(get(hEditComp1,'string'))));
            
            if dosyguidata.comparepeaks(1)>length(dosydata.freqs)
                dosyguidata.comparepeaks(1)=length(dosydata.freqs);
                set(hEditComp1,'String',num2str(dosyguidata.comparepeaks(1)))
            end
            if dosyguidata.comparepeaks(1)<0
                dosyguidata.comparepeaks(1)=1;
                set(hEditComp1,'String',num2str(dosyguidata.comparepeaks(1)))
            end
        end
        guidata(hMainFigure,dosyguidata);
    end
    function EditComp2_Callback(source,eventdata) %#ok<INUSD>
        dosyguidata=guidata(hMainFigure);
        if isnan((str2double(get(hEditComp2,'string'))))
            disp('Must be a number')
            set(hEditComp2,'String',num2str(dosyguidata.comparepeaks(2)))
        else
            dosyguidata.comparepeaks(2)=...
                round((str2double(get(hEditComp1,'string'))));
            
            if dosyguidata.comparepeaks(2)>length(dosydata.freqs)
                dosyguidata.comparepeaks(2)=length(dosydata.freqs);
                set(hEditComp1,'String',num2str(dosyguidata.comparepeaks(2)))
            end
            if dosyguidata.comparepeaks(2)<0
                dosyguidata.comparepeaks(2)=1;
                set(hEditComp1,'String',num2str(dosyguidata.comparepeaks(2)))
            end
        end
        guidata(hMainFigure,dosyguidata);
    end
    function EditComp3_Callback(source,eventdata) %#ok<INUSD>
        dosyguidata=guidata(hMainFigure);
        if isnan((str2double(get(hEditComp3,'string'))))
            disp('Must be a number')
            set(hEditComp3,'String',num2str(dosyguidata.comparepeaks(3)))
        else
            dosyguidata.comparepeaks(3)=...
                round((str2double(get(hEditComp3,'string'))));
            
            if dosyguidata.comparepeaks(3)>length(dosydata.freqs)
                dosyguidata.comparepeaks(3)=length(dosydata.freqs);
                set(hEditComp3,'String',num2str(dosyguidata.comparepeaks(3)))
            end
            if dosyguidata.comparepeaks(3)<0
                dosyguidata.comparepeaks(3)=1;
                set(hEditComp3,'String',num2str(dosyguidata.comparepeaks(3)))
            end
        end
        guidata(hMainFigure,dosyguidata);
    end
    function EditComp4_Callback(source,eventdata) %#ok<INUSD>
        dosyguidata=guidata(hMainFigure);
        if isnan((str2double(get(hEditComp4,'string'))))
            disp('Must be a number')
            set(hEditComp4,'String',num2str(dosyguidata.comparepeaks(4)))
        else
            dosyguidata.comparepeaks(4)=...
                round((str2double(get(hEditComp4,'string'))));
            
            if dosyguidata.comparepeaks(4)>length(dosydata.freqs)
                dosyguidata.comparepeaks(4)=length(dosydata.freqs);
                set(hEditComp4,'String',num2str(dosyguidata.comparepeaks(4)))
            end
            if dosyguidata.comparepeaks(4)<0
                dosyguidata.comparepeaks(4)=1;
                set(hEditComp4,'String',num2str(dosyguidata.comparepeaks(4)))
            end
        end
        guidata(hMainFigure,dosyguidata);
    end
    function EditComp5_Callback(source,eventdata) %#ok<INUSD>
        dosyguidata=guidata(hMainFigure);
        if isnan((str2double(get(hEditComp5,'string'))))
            disp('Must be a number')
            set(hEditComp5,'String',num2str(dosyguidata.comparepeaks(5)))
        else
            dosyguidata.comparepeaks(5)=...
                round((str2double(get(hEditComp5,'string'))));
            
            if dosyguidata.comparepeaks(5)>length(dosydata.freqs)
                dosyguidata.comparepeaks(5)=length(dosydata.freqs);
                set(hEditComp5,'String',num2str(dosyguidata.comparepeaks(5)))
            end
            if dosyguidata.comparepeaks(5)<0
                dosyguidata.comparepeaks(5)=1;
                set(hEditComp5,'String',num2str(dosyguidata.comparepeaks(5)))
            end
        end
        guidata(hMainFigure,dosyguidata);
    end
    function CheckComp1_Callback(source,eventdata)  %#ok<INUSD>
        if get(hCheckComp1,'Value')
            set(hEditComp1,'Enable','On')
        else
            set(hEditComp1,'Enable','Off')
        end
        guidata(hMainFigure,dosyguidata);
    end
    function CheckComp2_Callback(source,eventdata)  %#ok<INUSD>
        if get(hCheckComp2,'Value')
            set(hEditComp2,'Enable','On')
        else
            set(hEditComp2,'Enable','Off')
        end
        guidata(hMainFigure,dosyguidata);
    end
    function CheckComp3_Callback(source,eventdata)  %#ok<INUSD>
        if get(hCheckComp3,'Value')
            set(hEditComp3,'Enable','On')
        else
            set(hEditComp3,'Enable','Off')
        end
        guidata(hMainFigure,dosyguidata);
    end
    function CheckComp4_Callback(source,eventdata)  %#ok<INUSD>
        if get(hCheckComp4,'Value')
            set(hEditComp4,'Enable','On')
        else
            set(hEditComp4,'Enable','Off')
        end
        guidata(hMainFigure,dosyguidata);
    end
    function CheckComp5_Callback(source,eventdata)  %#ok<INUSD>
        if get(hCheckComp5,'Value')
            set(hEditComp5,'Enable','On')
        else
            set(hEditComp5,'Enable','Off')
        end
        guidata(hMainFigure,dosyguidata);
    end
    function CheckOrig_Callback(source,eventdata)  %#ok<INUSD>
        
        guidata(hMainFigure,dosyguidata);
    end
    function CheckFitted_Callback(source,eventdata)  %#ok<INUSD>
        
        guidata(hMainFigure,dosyguidata);
    end
    function CheckResid_Callback(source,eventdata)  %#ok<INUSD>
        if get(hCheckResid,'Value')
            dosyguidata.comparetype(3)=1;
        else
            dosyguidata.comparetype(3)=0;
        end
        guidata(hMainFigure,dosyguidata);
    end
    function ButtonCompareResid_Callback(source,eventdata) %#ok<INUSD>
        dosyguidata=guidata(hMainFigure);
        
        %Check data type
        if get(hCheckOrig,'Value')
            dosyguidata.comparetype(1)=1;
        else
            dosyguidata.comparetype(1)=0;
        end
        if get(hCheckFitted,'Value')
            dosyguidata.comparetype(2)=1;
        else
            dosyguidata.comparetype(2)=0;
        end
        if get(hCheckResid,'Value')
            dosyguidata.comparetype(3)=1;
        else
            dosyguidata.comparetype(3)=0;
        end
        
        
        %Check peak numbers
        if get(hCheckComp1,'Value')
            dosyguidata.comparepeaks(1)=str2double(get(hEditComp1,'String'));
            dosyguidata.compareuse(1)=1;
        else
            dosyguidata.compareuse(1)=0;
        end
        if get(hCheckComp2,'Value')
            dosyguidata.comparepeaks(2)=str2double(get(hEditComp2,'String'));
            dosyguidata.compareuse(2)=1;
        else
            dosyguidata.compareuse(2)=0;
        end
        if get(hCheckComp3,'Value')
            dosyguidata.comparepeaks(3)=str2double(get(hEditComp3,'String'));
            dosyguidata.compareuse(3)=1;
        else
            dosyguidata.compareuse(3)=0;
        end
        if get(hCheckComp4,'Value')
            dosyguidata.comparepeaks(4)=str2double(get(hEditComp4,'String'));
            dosyguidata.compareuse(4)=1;
        else
            dosyguidata.compareuse(4)=0;
        end
        if get(hCheckComp5,'Value')
            dosyguidata.comparepeaks(5)=str2double(get(hEditComp5,'String'));
            dosyguidata.compareuse(5)=1;
        else
            dosyguidata.compareuse(5)=0;
        end
        guidata(hMainFigure,dosyguidata);
        
        dosyresidualcompare(dosydata,dosyguidata);
    end
    function EditFlip_Callback(source,eventdata)                       %#ok
        dosyguidata.xlim = get(hDOSY, 'Xlim');
        dosyguidata.ylim = get(hDOSY, 'Ylim');
        dosyguidata=guidata(hMainFigure);
        dosyguidata.residpeak=round(str2double(get(hEditFlip,'String')));
        if dosyguidata.residpeak>length(dosydata.freqs)
            dosyguidata.residpeak=length(dosydata.freqs);
            set(hEditFlip,'String',num2str(dosyguidata.residpeak))
        end
        if dosyguidata.residpeak<1
            dosyguidata.residpeak=1;
            set(hEditFlip,'String',num2str(dosyguidata.residpeak))
        end
        guidata(hMainFigure,dosyguidata);
        PlotResidual(dosyguidata.residpeak);
    end
    function ButtonFlipPlus_Callback(source,eventdata) %#ok
        dosyguidata.xlim = get(hDOSY, 'Xlim');
        dosyguidata.ylim = get(hDOSY, 'Ylim');
        dosyguidata=guidata(hMainFigure);
        dosyguidata.residpeak=dosyguidata.residpeak+1;
        if dosyguidata.residpeak>length(dosydata.freqs)
            dosyguidata.residpeak=length(dosydata.freqs);
            set(hEditFlip,'String',num2str(dosyguidata.residpeak))
        end
        if dosyguidata.residpeak<1
            dosyguidata.residpeak=1;
            set(hEditFlip,'String',num2str(dosyguidata.residpeak))
        end
        set(hEditFlip,'String',num2str(dosyguidata.residpeak))
        guidata(hMainFigure,dosyguidata);
        PlotResidual(dosyguidata.residpeak);
    end
    function ButtonFlipMinus_Callback(source,eventdata)%#ok
        dosyguidata.xlim = get(hDOSY, 'Xlim');
        dosyguidata.ylim = get(hDOSY, 'Ylim');
        dosyguidata=guidata(hMainFigure);
        dosyguidata.residpeak=dosyguidata.residpeak-1;
        if dosyguidata.residpeak>length(dosydata.freqs)
            dosyguidata.residpeak=length(dosydata.freqs);
            set(hEditFlip,'String',num2str(dosyguidata.residpeak))
        end
        if dosyguidata.residpeak<1
            dosyguidata.residpeak=1;
            set(hEditFlip,'String',num2str(dosyguidata.residpeak))
        end
        set(hEditFlip,'String',num2str(dosyguidata.residpeak))
        guidata(hMainFigure,dosyguidata);
        PlotResidual(dosyguidata.residpeak);
        
    end
%% -----Utility functions -----------------------------------------------
    function PlotContour()
        dosyguidata=guidata(hMainFigure);
        hxkResidLine=findobj(hResid,'tag','xkResidLine');
        delete(hxkResidLine);
        rResidLine=findobj(hResid,'tag','rResidLine');
        delete(rResidLine);
        bkResidLine=findobj(hResid,'tag','bResidLine');
        delete(bkResidLine);

        axes(hDOSY)
        get(hMainFigure,'CurrentAxes');
        set(hDOSY,'Visible','On')
        set(hResid,'Visible','Off')
        
        xxx=xlim();
        yyy=ylim();
        
        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            %DOSY type data
            [xax,yax]=meshgrid(dosydata.Ppmscale,dosydata.Dscale);
            
            [~,hContour1]=contour(hDOSY,xax,yax,(dosydata.DOSY.*(dosydata.DOSY>0))',dosyguidata.clvl,'b','Tag','DOSYContour1');
            if dosyguidata.plotopt==0  
                imagesc(hDOSY,dosydata.Ppmscale,dosydata.Dscale,dosydata.DOSY','Tag','DOSYImage');
                colormap(dosyguidata.immap)
            end
            set(hContour1,'LineWidth',1)
            if dosydata.DOSY<0
                hold(hDOSY,'on')
                [~,hContour2]=contour(hDOSY,xax,yax,(dosydata.DOSY.*(dosydata.DOSY<0))',dosyguidata.clvl,'color',[1 0.5 0],'Tag','DOSYContour2');
                hold(hDOSY,'off')
                
                dosyguidata.hContour2=hContour2;
                set(hContour2,'LineWidth',1)
            end
            ylabel(hDOSY,'\bf Diffusion coefficient / 10^{-10} m^{2} s^{-1}');
            
            
            
            
        elseif strcmp(dosydata.type,'t1/t2')
            %relaxation data
            [xax,yax]=meshgrid(dosydata.Ppmscale,dosydata.T1scale);
            
            [~,hContour1]=contour(hDOSY,xax,yax,dosydata.T1',dosyguidata.clvl,'b','Tag','DOSYContour1');
            
            if dosyguidata.plotopt==0 
                imagesc(hDOSY,dosydata.Ppmscale,dosydata.T1scale,dosydata.T1','Tag','DOSYImage');
                colormap(dosyguidata.immap)
            end
            
            if dosydata.FitType == 3 
                ylabel(hDOSY,'\bf T_2 coefficient (s)');
            else
                ylabel(hDOSY,'\bf T_1 coefficient (s)');
            end

            dosyguidata.hContour1=hContour1;
            dosyguidata.hContour2=[];
            
            set(hContour1,'LineWidth',1)
            
        else
            error('Unknown data type')
        end
        
        set(hDOSY,'LineWidth',1);
        set(hDOSY,'Ydir','reverse');
        set(hDOSY,'Xdir','reverse');
        xlabel(hDOSY,'\bf Chemical shift (ppm)');
        
        xlim(xxx);
        
        PlotSpectrum();
        set(hDOSY, 'Ylim',yyy,'Xlim',xxx);
        linkaxes([hDOSY hSpec],'x');
        guidata(hMainFigure,dosyguidata);
        
        
    end
    function PlotSpectrum()
        axes(hSpec);
        hResidLine=findobj(hSpec,'tag','ResidLine');
        delete(hResidLine);
        xxx=xlim();
        hSpecPlot=plot(hSpec,dosydata.Ppmscale, real(dosydata.Spectrum),'-b','tag','Projection');
        axis(hSpec,'tight')
        axis(hSpec,'off')
        set(hSpec,'Xdir','reverse');
        set(hSpecPlot,'LineWidth',1)
        set(hSpecPlot,'Color','black')
        xlim(xxx);
   
        
        if CurrDigFac >1
       
        yyy=ylim();
        text(xxx(2)*0.2, yyy(2)*0.9, ["Low digitisation for faster plotting,", "not suitable for publication"],'color',[0.5 .5 0.5])
        end
        
    end
    function PlotResidual(npeak)
        dosyguidata.xlim = get(hDOSY, 'Xlim');
        dosyguidata.ylim = get(hDOSY, 'Ylim');
        guidata(hMainFigure,dosyguidata);
        linkaxes([hDOSY hSpec],'off')
        axes(hResid)
        set(hResid,'Visible','On')
        set(hDOSY,'Visible','Off')
        get(hMainFigure,'CurrentAxes');
        norm_max=max([max(dosydata.ORIGINAL(npeak,:)) max(dosydata.FITTED(npeak,:))]);
        
        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            %DOSY type data
            xdata=dosydata.Gzlvl;
            xlabel('\fontname{ariel} \bf Squared gradient amplitude /(T^2 m^{-2})')
        elseif strcmp(dosydata.type,'t1/t2')
            %relaxation data
            xdata=dosydata.Tau;
            xlabel('\fontname{ariel} \bf Time / s')
        else
            error('Unknown data type')
        end
        
        % Main residual plot
        % Fitting plot
        plot(xdata,(dosydata.ORIGINAL(npeak,:)./norm_max),'xk','LineWidth',1.5,'tag','xkResidLine')
        hold on
        if isfield(dosydata,'FITTED_CALC')
            plot(dosydata.X_CALC,(dosydata.FITTED_CALC(npeak,:)./norm_max),'--r','LineWidth',1,'tag','rResidLine');
        else
            plot(xdata,(dosydata.FITTED(npeak,:)./norm_max),'--r','LineWidth',1,'tag','rResidLine')
        end
        plot(xdata,(dosydata.RESIDUAL(npeak,:)./norm_max),'b--o','LineWidth',1,'tag','bResidLine')
        
        set(gca,'LineWidth',1);
        axis('tight')
        ylim('auto')
        if min(dosydata.FITTED(npeak,:)./norm_max) <0
            ylim([min(dosydata.FITTED(npeak,:)./norm_max)*1.1 1]);
        else
            ylim([-0.1 1]);
        end
        
        % Text plot 
        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            %diffusion data
            ylabel('\fontname{ariel} \bf Amplitude')
            str=['\fontname{ariel} \rm Peak number: ' num2str(npeak)];
            text(0.75*max(xdata),0.55,str);
            str=['\fontname{ariel} \rm Frequency: ' num2str(dosydata.freqs(npeak),3) ' ppm'];
            text(0.75*max(xdata),0.48,str);
            nexp=length(find(dosydata.FITSTATS(npeak,:)))/4;
            str=['\fontname{ariel} \rm Number of exponentials: ' num2str(nexp,1)];
            text(0.75*max(xdata),0.41,str);
            str='\color{black} \rm x  Original';
            text(0.75*max(xdata),0.85,str);
            str='\color{red} \rm - -  Fitted';
            text(0.75*max(xdata),0.78,str);
            str='\color{blue} \rm - -  Residual';
            text(0.75*max(xdata),0.71,str);
            
        elseif strcmp(dosydata.type,'t1/t2')
            %relaxation data
            if dosydata.FitType == 1 % T1 data
                ylabel('\fontname{ariel} \bf Amplitude')
                str=['\fontname{ariel} \rm Peak number: ' num2str(npeak)];
                text(0.75*max(xdata),-0.2,str);
                str=['\fontname{ariel} \rm Frequency: ' num2str(dosydata.freqs(npeak),3) ' ppm'];
                text(0.75*max(xdata),-0.3,str);
                str='\color{black} \rm x  Original';
                text(0.75*max(xdata),0.4,str);
                str='\color{red} \rm - -  Fitted';
                text(0.75*max(xdata),0.3,str);
                str='\color{blue} \rm - -  Residual';
                text(0.75*max(xdata),0.2,str);
            else % T1 inv. rec. and T2 data
                ylabel('\fontname{ariel} \bf Amplitude')
                str=['\fontname{ariel} \rm Peak number: ' num2str(npeak)];
                text(0.75*max(xdata),0.52,str);
                str=['\fontname{ariel} \rm Frequency: ' num2str(dosydata.freqs(npeak),3) ' ppm'];
                text(0.75*max(xdata),0.44,str);
                nexp=length(find(dosydata.FITSTATS(npeak,:)))/4;
                str=['\fontname{ariel} \rm Number of exponentials: ' num2str(nexp,1)];
                text(0.75*max(xdata),0.36,str);
                str='\color{black} \rm x  Original';
                text(0.75*max(xdata),0.85,str);
                str='\color{red} \rm - -  Fitted';
                text(0.75*max(xdata),0.77,str);
                str='\color{blue} \rm - -  Residual';
                text(0.75*max(xdata),0.69,str);
            end
        end

        % X-axis label
        if strcmp(dosydata.type,'dosy')|| strcmp(dosydata.type,'ILT')...
                || strcmp(dosydata.type,'locodosy')|| strcmp(dosydata.type,'fdm/rrt')
            %DOSY type data
            xlabel('\fontname{ariel} \bf Squared gradient amplitude /(T^2 m^{-2})')
        elseif strcmp(dosydata.type,'t1/t2')
            %relaxation data
            xlabel('\fontname{ariel} \bf Time / s')
        else
            error('Unknown data type')
        end
        
        hold off
        
        
        % Plot line in the projection to know which peak is shown
        PlotSpectrum()
        axes(hSpec)
        line([dosydata.freqs(npeak) dosydata.freqs(npeak)], ylim,'Color', 'r','tag','ResidLine')
    end
end