function [] = segwe_gnat()
%% --- SEGWE D/MW Calculator Info version 2.0 (updated 30/10/2018)
% 
%      This standalone GUI can be used to determine molecular diffusion
%      coefficient and hydrodynamic radius from molecular weight (and
%      vice-versa).
%      The recommended method is "Stokes-Einstein Gierer-Wirtz Estimation"
%      (or SEGWE for short) - which is based on viscosity scaling and
%      effective density for the solute, but the classial "Stokes-Einstein"
%      method is also available.
%      Temperature is set by user, and a large list of solvent can be used.
%
%      For more information DOI: 10.1021/acs.analchem.7b05032
%
%% --- Copyright  2015  <Guilherme Dal Poggetto dalpgt@unicamp.br>
%
%   http://nmr.chemistry.manchester.ac.uk/
%   The University of Manchester
%
%   This program is free software; you can redistribute it and/or modify it
%   under the terms of the GNU General Public License as published by the
%   Free Software Foundation; either version 2 of the License, or (at your
%   option) any later version.
%
%   This program is distributed in the hope that it will be useful, but
%   WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%   General Public License for more details.
%
%   You should have received a copy of the GNU General Public License along
%   with this program; if not,  see <http://www.gnu.org/licenses/> or write
%   to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
%   Boston, MA 02110-1301 USA.
%
%% --- Abreviations in the code
%   MW = molecular weight of the solute (g mol-1)
%   MWs = MW of the solvent (g mol-1)
%   form = molecular formula of the solvent;
%   n = viscosity of the solvent(kg m-1 s-1)
%   D = diffusion coefficient of the solute (m2 s-1)
%   kB = Boltzmann constant (m2 kg s-2 K-1)
%   NA = Avogadro's number
%   peff = effective viscosity taken from reference 1 (kg m-3)
%   p_bulk = density of the bulk material(1e6 g cm-3)
%   packing = packing fraction (0.64 for liquids and 0.74 for solids)
%   a = slope in the Arrenhius aproximation of the viscosity
%   b = intercept in the Arrenhius aproximation of the viscosity
%   
%% --- Defaults

    %   Constants
    packing = 0.64;
    kB = 1.380644e-23;
    NA = 6.02214e23;
    T = 298.15;
    info=0;

    %   Initial Condition - Acetone d6
    p_bulk = 1.05;
    MWs=64.12;
    n=0.0003258;
    a=-10.935;
    b=866.334;
    p=-0.1;
    q=0.5;
    peff = 627;
    

    %   Initial Values on display
    argument = 100;
    answer = 30.80;
    passive = 398.38;

    %   Calculate data
    dosydata.temperature=T;
    dosydata.argument=argument;
    dosydata.packing=packing;
    dosydata.viscositydisplay=n*1000;
    dosydata.answer=answer;
    dosydata.passive=passive;
    dosydata.density=p_bulk;
    dosydata.MWs=MWs;
    dosydata.peff=peff;
%% --- GUI Interface
    %   Main
    hMainFigure = figure(...
        'Units','pixels',...
        'MenuBar','none',...
        'Name','SEGWE D/MW Calculator',...
        'NumberTitle','Off',...
        'OuterPosition',[250 250 700 500 ],...
        'Visible','off');
    guidata(hMainFigure,dosydata);
    
    %   GUI Title
    uicontrol(...
        'Parent',hMainFigure,...
        'Style','text',...
        'Units','Normalized',...
        'FontSize',28,...
        'FontWeight','bold',...
        'horizontalalignment','l',...
        'Position',[0.18 0.57 1 0.4 ],...
        'String','SEGWE D/MW Calculator' );

    %   Conditions
    hPanelMethod=uipanel(...
        'Parent',hMainFigure,...
        'Units','Normalized',...
        'Title','Conditions',...
        'FontWeight','bold',...
        'FontSize',13,...
        'TitlePosition','centertop',...
        'ForegroundColor','Black',...
        'Position',[0.05 0.02 0.45 0.84]);
    
    %   Method text
    uicontrol(...
        'Parent',hPanelMethod,...
        'Style','text',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'horizontalalignment','l',...
        'Position',[0.05 0.55 1 0.4 ],...
        'String','Method' );

    %   Method Popup list
    hEstimationpopup=uicontrol(...
        'Parent',hPanelMethod,...
        'Style','popupmenu',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'horizontalalignment','l',...
        'Position',[0.5 0.885 0.45 0.08 ],...
        'String','Stokes-Einstein-Gierer-Wirtz|Stokes-Einstein',...
        'CallBack', {@method_Callback});

    %   Solvent text
    uicontrol(...
        'Parent',hPanelMethod,...
        'Style','text',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'horizontalalignment','l',...
        'Position',[0.05 0.41 0.45 0.4 ],...
        'String','Solvent' );

    %   Solvent Popup list
    hSolventpopup=uicontrol(...
        'Parent',hPanelMethod,...
        'Style','popupmenu',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'horizontalalignment','l',...
        'Position',[0.5 0.43 0.45 0.4 ],...
        'String',...
        'Acetone-d6|Acetonitrile|Acetonitrile-d3|Benzene|Benzene-d6|Chloroform|Chloroform-d1|Cyclohexane|Cyclohexane-d12|Cyclopentane|Dioxane|Dioxane-d8|DMF|DMF-d7|DMSO|DMSO-d6|n-Dodecane|Ethanol|Ethanol-d6|Fluorobenzene|n-Hexane|Methanol|Methanol-d3|Methanol-d4|Nitromethane-d3|Pentanol|Pyridine-d5|n-Tetradecane|THF|THF-d8|Toluene|Toluene-d8|Water-d2|Water|User defined',...
        'CallBack', {@solvent_Callback});

    %   Temperature Text    
    uicontrol(...
        'Parent',hPanelMethod,...
        'Style','text',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'horizontalalignment','l',...
        'Position',[0.05 0.275 0.45 0.4 ],...
        'String','Temperature / K' );

    %   Temperature Input
    hTemperature=uicontrol(...
        'Parent',hPanelMethod,...
        'Style','edit',...
        'Visible','On',...
        'BackgroundColor','w',...
        'Units','Normalized',...
        'String',sprintf('%.2f',T),...
        'FontSize',10,...
        'Enable','On',...
        'Visible','On',...
        'Max',1,...
        'Position',[0.5 0.616 0.45 0.08],...
        'CallBack', {@temperature_Callback});

    %   Viscosity Display
    uicontrol(...
        'Parent',hPanelMethod,...
        'Style','text',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'horizontalalignment','l',...
        'Position',[0.05 0.135 0.45 0.4 ],...
        'String','Estimated Viscosity / (mPa s)' );

    %   Viscosity Output
    hViscosity=uicontrol(...
        'Parent',hPanelMethod,...
        'Style','edit',...
        'Visible','On',...
        'BackgroundColor','w',...
        'Units','Normalized',...
        'FontSize',10,...
        'Enable','Off',...
        'String',sprintf('%.4f',dosydata.viscositydisplay),...
        'Max',1,...
        'Position',[0.5 0.463 0.45 0.08]);
    
    %   Bulk density text
    uicontrol(...
        'Parent',hPanelMethod,...
        'Style','text',...
        'Units','Normalized',...
        'FontWeight','normal',...
        'Max',2,...
        'FontSize',10,...
        'horizontalalignment','l',...
        'Position',[0.05 -0.015 0.45 0.4 ],...
        'TooltipString','SEGWE uses effective viscosity and SE uses user defined value',...
        'String','Solute density / (g cm-3))' );
    
    %   Bulk density Input
    hdensity=uicontrol(...
        'Parent',hPanelMethod,...
        'Style','edit',...
        'Visible','On',...
        'BackgroundColor','w',...
        'Units','Normalized',...
        'Enable','Off',...
        'FontSize',10,...
        'String',sprintf('%.4f',0.627),...
        'Max',1,...
        'Position',[0.5 0.315 0.45 0.08],...
        'TooltipString','SEGWE uses effective viscosity and SE uses user defined value',...
        'CallBack', {@density_Callback});
    
    %   Packing fraction text
    uicontrol(...
        'Parent',hPanelMethod,...
        'Style','text',...
        'Units','Normalized',...
        'FontWeight','normal',...
        'Max',2,...
        'FontSize',10,...
        'horizontalalignment','l',...
        'TooltipString','SEGWE uses 1 and SE uses either 0.64 (for liquids) or 0.74 (for solids)',...
        'Position',[0.05 -0.165 0.45 0.4 ],...
        'String','Packing fraction');
    
    %   Packing fraction Input
    hPacking=uicontrol(...
        'Parent',hPanelMethod,...
        'Style','edit',...
        'Visible','On',...
        'BackgroundColor','w',...
        'Units','Normalized',...
        'Enable','Off',...
        'FontSize',10,...
        'String',sprintf('%.2f',1.00),...
        'Max',1,...
        'CallBack', {@packing_Callback},...
        'TooltipString','SEGWE uses 1 and SE uses either 0.64 (for liquids) or 0.74 (for solids)',...
        'Position',[0.5 0.17 0.45 0.08]);

    %	Calculate button
    uicontrol(...
        'Parent',hPanelMethod,...
        'Style','PushButton',...
        'Visible','On',...
        'Units','Normalized',...
        'Enable','On',...
        'FontSize',13,...
        'String','Calculate',...
        'TooltipString','Calculate property choosen in "Estimation Mode"',...
        'Position',[0.1 0.02 0.45 0.1 ],...
        'CallBack', {@calculate_Callback});
    
    %	Confidence button
    hCIButton=uicontrol(...
        'Parent',hPanelMethod,...
        'Style','PushButton',...
        'Visible','On',...
        'Units','Normalized',...
        'Enable','On',...
        'FontSize',13,...
        'String','Confidence',...
        'TooltipString','Show 95% confidence interval of current output',...
        'Position',[0.6 0.02 0.3 0.1 ],...
        'CallBack', {@confidence_Callback});

    %   Estimation Mode radios
    hEstimation_group=uibuttongroup(...
        'Parent',hMainFigure,...
        'Units','Normalized',...
        'Title','Estimation Mode',...
        'FontWeight','bold',...
        'FontSize',13,...
        'TitlePosition','centertop',...
        'ForegroundColor','Black',...
        'SelectionChangeFcn',{@estimation_Callback},...
        'Position',[0.51 0.521 0.45 0.34]);
    uicontrol(...
        'Parent',hEstimation_group,...
        'Style','Radio',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'horizontalalignment','l',...
        'Position',[0.25 0.78 0.9 0.2 ],...
        'String','MW to D and rH',...
        'TooltipString','Molecular Weight to Diffusion Coefficient and Hydrodynamic radius');
    uicontrol(...
        'Parent',hEstimation_group,...
        'Style','Radio',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'horizontalalignment','l',...
        'Position',[0.25 0.45 0.9 0.2 ],...
        'TooltipString','Diffusion Coefficient to Molecular Weight and Hydrodynamic radius',...
        'String','D to MW and rH');
    uicontrol(...
        'Parent',hEstimation_group,...
        'Style','Radio',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'horizontalalignment','l',...
        'Position',[0.25 0.12 0.9 0.2 ],...
        'HandleVisibility', 'on',...
        'TooltipString','Hydrodynamic radius to Diffusion Coefficient and Molecular Weight',...
        'String','rH to D and MW');

    %   Input/Output panel
    hPanelInOut=uipanel(...
        'Parent',hMainFigure,...
        'Units','Normalized',...
        'Title','Input/Output',...
        'Visible','On',...
        'FontWeight','bold',...
        'FontSize',13,...
        'TitlePosition','centertop',...
        'ForegroundColor','Black',...
        'Position',[0.51 0.025 0.45 0.49]);
    
    %   Argument text
    uicontrol(...
        'Parent',hPanelInOut,...
        'Style','text',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'Max',2,...
        'TooltipString','Both SEGWE and SE assumes spherical particles',...
        'horizontalalignment','l',...
        'Position',[0.05 0.5 0.38 0.4 ],...
        'String','Molecular Weight / (g mol-1)' );

    %   Argument Input
    hArgument=uicontrol(...
        'Parent',hPanelInOut,...
        'Style','edit',...
        'Visible','On',...
        'BackgroundColor','w',...
        'Units','Normalized',...
        'FontSize',11,...
        'FontWeight','bold',...
        'Enable','On',...
        'String',sprintf('%.2f',argument),...
        'Max',1,...
        'Position',[0.51 0.75 0.45 0.15],...
        'CallBack', {@argument_Callback});
    
    %   Passive text
    uicontrol(...
        'Parent',hPanelInOut,...
        'Style','text',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'Max',2,...
        'TooltipString','Both SEGWE and SE assumes spherical particles',...
        'horizontalalignment','l',...
        'Position',[0.05 0.186 0.46 0.4 ],...
        'String','Hydrodynamic radius / (10^-12 m)');

    %   Passive Output
    hPassive=uicontrol(...
        'Parent',hPanelInOut,...
        'Style','edit',...
        'Visible','On',...
        'BackgroundColor','w',...
        'Units','Normalized',...
        'FontSize',11,...
        'FontWeight','normal',...
        'Enable','inactive',...
        'String',sprintf('%.2f',passive),...
        'Max',1,...
        'Position',[0.51 0.43 0.45 0.15]);
    
    %   Answer text
    uicontrol(...
        'Parent',hPanelInOut,...
        'Style','text',...
        'Units','Normalized',...
        'FontSize',10,...
        'FontWeight','normal',...
        'Max',2,...
        'TooltipString','Both SEGWE and SE assumes spherical particles',...
        'horizontalalignment','l',...
        'Position',[0.05 -0.14 0.43 0.4 ],...
        'String','Diffusion Coefficient / (10^-10 m2 s-1)');

    %   Answer Output
    hAnswer=uicontrol(...
        'Parent',hPanelInOut,...
        'Style','edit',...
        'Visible','On',...
        'BackgroundColor','w',...
        'Units','Normalized',...
        'FontSize',11,...
        'FontWeight','normal',...
        'Enable','inactive',...
        'String',sprintf('%.2f',answer),...
        'Max',1,...
        'Position',[0.51 0.11 0.45 0.15]);

    movegui(hMainFigure,'center');
    set(hMainFigure,'Visible','on');
%% --- Callbacks
%   Temperature callback
    function temperature_Callback(~,~)
        dosydata=guidata(hMainFigure);
        T=str2double(get(hTemperature,'String'));
        if T<0 || T==0 || isnan(T)
            errordlg('Temperature must be a positive number, in Kelvin')
            dosydata.temperature=298.15;
        else
            dosydata.temperature=T;
        end
        set(hTemperature,'String',sprintf('%.2f',dosydata.temperature))
        guidata(hMainFigure,dosydata);
        solvent_Callback()
    end

%	Argument Callback / Input
%   (either D,rH or MW, depending on the function)
    function argument_Callback(~,~)
        dosydata=guidata(hMainFigure);
        argument=str2double(get(hArgument,'String'));
        if argument<0 || argument==0
            errordlg('This must be a positive number')
            dosydata.argument=100;
            set(hArgument,'String',num2str(dosydata.argument))
        else
            dosydata.argument=argument;
        end
        set(hArgument,'String',sprintf('%.2f',dosydata.argument))
        set(hAnswer,'String','Press Calculate')
        set(hPassive,'String','Press Calculate')
        guidata(hMainFigure,dosydata);
    end

%	Packing Callback / Input
%   (used only in SE)
    function packing_Callback(~,~)
        dosydata=guidata(hMainFigure);
        packing=str2double(get(hPacking,'String'));
        if packing==0.64 || packing==0.74
            dosydata.packing=packing;
        else
            errordlg('Packing fraction of material is either 0.64 (for liquids) or 0.74 (for solids)')
            dosydata.packing=0.64;
            set(hPacking,'String',num2str(dosydata.packing))
        end
        set(hPacking,'String',sprintf('%.2f',dosydata.packing))
        set(hAnswer,'String','Press Calculate')
        set(hPassive,'String','Press Calculate')
        guidata(hMainFigure,dosydata);
    end

%	Density Callback / Input
%   (used only in SE)
    function density_Callback(~,~)
        dosydata=guidata(hMainFigure);
        p_bulk=str2double(get(hdensity,'String'));
        if p_bulk<0 || p_bulk==0
            errordlg('This must be a positive number')
            dosydata.density=1.050;
            set(hdensity,'String',num2str(dosydata.density))
        else
            dosydata.density=p_bulk;
        end
        set(hdensity,'String',sprintf('%.4f',dosydata.density))
        set(hAnswer,'String','Press Calculate')
        set(hPassive,'String','Press Calculate')
        guidata(hMainFigure,dosydata);
    end

%	Functions Callback
%   (Chooses between mw_2_d, rH_2_mw or d_2_mw)
    function estimation_Callback (~,~)
        dosydata=guidata(hMainFigure);
        switch get(get(hEstimation_group,'SelectedObject'),'String')
            case 'D to MW and rH'
                set(hAnswer,'Position',[0.51 0.75 0.45 0.15]); %MW
                set(hArgument,'Position',[0.51 0.11 0.45 0.15]); %D
                set(hPassive,'Position',[0.51 0.43 0.45 0.15]); %Hr
                set(hArgument,'String',[])
                set(hAnswer,'String','Press Calculate')
                set(hPassive,'String','Press Calculate')
                dosydata.argument=0;
            case 'MW to D and rH'
                set(hAnswer,'Position',[0.51 0.11 0.45 0.15]); %D
                set(hArgument,'Position',[0.51 0.75 0.45 0.15]); %MW
                set(hPassive,'Position',[0.51 0.43 0.45 0.15]); %Hr
                set(hArgument,'String',[])
                set(hAnswer,'String','Press Calculate')
                set(hPassive,'String','Press Calculate')
                dosydata.argument=0;
            case 'rH to D and MW'
                set(hAnswer,'Position',[0.51 0.11 0.45 0.15]); %D
                set(hArgument,'Position',[0.51 0.43 0.45 0.15]); %Hr
                set(hPassive,'Position',[0.51 0.75 0.45 0.15]); %MW
                set(hArgument,'String',[])
                set(hAnswer,'String','Press Calculate')
                set(hPassive,'String','Press Calculate')
                dosydata.argument=0;
        end
        set(hAnswer,'String','Press Calculate')
        set(hPassive,'String','Press Calculate')
        guidata(hMainFigure,dosydata);
    end

%	Solvent Callback
%   (The solvent choose in the popup menu will them select values of 'a',
%   'b', 'peff', 'n' and 'MWs' to be used in the equations)
    function solvent_Callback(~,~)
        dosydata=guidata(hMainFigure);
        item = get(hSolventpopup,'Value'); % list entry
        list = get(hSolventpopup,'String'); % list of solvents
        solvent = strtrim(list(item,:)); % retrieve solvent as string
        
        %Arrenhius aproximation of the viscosity, dependent of temperature
        %ploting 'log(n) vs. 1/(T)'
        %Where R is the 'gas constant' = 8.314 (J mol-1 K-1)
        %a = (intercept)
        %b = (slope)
        %form = chemical formula
        switch solvent
            case 'Acetone-d6' %form = '(CD3)2CO'
                MWs = 64.12;
                a = -10.935;
                b = 866.334;
                peff=627;
            case 'Acetonitrile' %form = 'CH3CN'
                MWs = 41.05;
                a = -10.9363;
                b = 899.42;
                peff=627;
            case 'Acetonitrile-d3' %form = 'CD3CN'
                MWs = 44.07;
                a = -10.9362;
                b = 894.669;
                peff=627;
            case 'Benzene' %form = 'C6H6'
                MWs = 78.11;
                a = -11.6111;
                b = 1251.94;
                peff=627;
            case 'Benzene-d6' %form = 'C6D6'
                MWs = 84.15;
                a = -11.5689;
                b = 1256.41;
                peff=627;
            case 'Chloroform' %form = 'CHCl3'
                MWs = 119.47;
                a = -10.459;
                b = 877.553;
                peff=627;
            case 'Chloroform-d1' %form = 'CDCl3'
                MWs = 120.38;
                a = -10.462;
                b = 877.553;
                peff=627;
            case 'Cyclohexane' %form = 'C6H12'
                MWs = 84.16;
                a = -12.0241;
                b = 1492.1;
                peff=627;
            case 'Cyclohexane-d12' %form = 'C6D12'
                MWs = 96.23;
                a = -11.9966;
                b = 1504.43;
                peff=627;
            case 'Cyclopentane' %form = 'C5H10'
                MWs = 70.1;
                a = -10.8545;
                b = 916.833;
                peff=627;
            case 'Dioxane' %form = 'C4H8O2'
                MWs = 88.11;
                a = -11.594;
                b = 1450.38;
                peff=627;
            case 'Dioxane-d8' %form = 'C4D8O2'
                MWs = 96.15;
                a = -11.8432;
                b = 1549.19;
                peff=627;
            case 'DMF' %form = 'HCON(CH3)2'
                MWs = 73.09;
                a = -10.8464;
                b = 1113.68;
                peff=627;
            case 'DMF-d7' %form = 'DCON(CD3)2'
                MWs = 80.14;
                a = -10.4328;
                b = 1104.48;
                peff=627;
            case 'DMSO' %form = '(CH3)2SO'
                MWs = 78.13;
                a = -11.6929;
                b = 1631.34;
                peff=627;
            case 'DMSO-d6' %form = '(CD3)2SO'
                MWs = 84.17;
                a = -11.9712;
                b = 1741.84;
                peff=627;
            case 'n-Dodecane' %form = 'C12H26'
                MWs = 170.33;
                a = -11.4967;
                b = 1458.05;
                peff=627;
            case 'Ethanol' %form = 'CH3CH2OH'
                MWs = 46.07;
                a = -12.2566;
                b = 1626.83;
                peff=627;
            case 'Ethanol-d6' %form = 'CD3CD2OD'
                MWs = 52.11;
                a = -12.4615;
                b = 1705.79;
                peff=627;
            case 'Fluorobenzene' %form = 'C6H5F'
                MWs = 96.1;
                a = -10.9673;
                b = 1036.43;
                peff=627;
            case 'n-Hexane' %form = 'C6H12'
                MWs = 86.18;
                a = -11.1161;
                b = 890.301;
                peff=627;
            case 'Methanol' %form = 'CH3OH'
                MWs = 32.04;
                a = -11.5566;
                b = 1202.6;
                peff=627;
            case 'Methanol-d3' %form = 'CD3OH'
                MWs = 35.063;
                a = -11.6196;
                b = 1246.5;
                peff=627;
            case 'Methanol-d4' %form = 'CD3OD'
                MWs= 36.07;
                a=-11.6611;
                b=1267.68;
                peff=627;
            case 'Nitromethane-d3' %form = 'CD3NO2'
                MWs = 64.06;
                a = -10.7876;
                b = 1023.31;
                peff=627;
            case 'Pentanol' %form = 'C5H12O'
                MWs = 88.15;
                a = -13.939;
                b = 2457.95;
                peff=627;
            case 'Pyridine-d5' %form = 'C5D5N'
                MWs = 84.13;
                a = -11.3107;
                b = 1296.32;
                peff=627;
            case 'n-Tetradecane' %form = 'CH3(CH2)12CH3'
                MWs = 198.39;
                a = -12.0256;
                b = 1745.36;
                peff=627;
            case 'THF' %form = 'C4H8O'
                MWs = 72.11;
                a = -10.3993;
                b = 811.96;
                peff=627;
            case 'THF-d8' %form = 'C4D8O'
                MWs = 80.16;
                a = -10.7209;
                b = 930.4;
                peff=627;
            case 'Toluene' %form = 'C6H5CH3'
                MWs = 92.14;
                a = -11.0772;
                b = 1067.67;
                peff=627;
            case 'Toluene-d8' %form = 'C6D5CD3'
                MWs = 100.19;
                a = -11.1049;
                b = 1099.36;
                peff=627;
            case 'Water-d2' %form = 'D2O'
                MWs = 20.03;
                a = -14.6094;
                b = 2329.54;
                peff=627;
            case 'Water' %form = 'H2O'
                MWs = 18.01;
                a = -14.0236;
                b = 2092.95;
                peff=627;
            case 'User defined'
                prompt = {'Solvent Viscosity:','Solvent Molecular weight:'};
                dlg_title = 'User defined';
                num_lines = 1;
                defaultans = {num2str(dosydata.viscositydisplay),num2str(dosydata.MWs)};
                answer = inputdlg(prompt,dlg_title,num_lines,defaultans);
                if isempty(answer)
                    n=dosydata.viscositydisplay/1000;
                    a=0;
                    b=dosydata.temperature*log(n);
                    MWs=dosydata.MWs;
                else
                    answer=str2double(answer);
                    n=answer(1);
                    a=0;
                    b=dosydata.temperature*log(n/1000);
                    MWs=answer(2);
                end
                peff=627;
            case' '
%%%%%%%%%%%%%% New solvent %%%%%%%%%%%%%
%           case 'new_solvent' %form = 'CxHyOzNn'
%                 MWs = 999.99;
%                 a = -99.9999;
%                 b = 9999.99;
%                 peff=627;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        end
        n=exp(a+(b/dosydata.temperature));
        dosydata.viscositydisplay=n*1000;
        dosydata.MWs=MWs;
        dosydata.peff=peff;
        set(hViscosity,'String',sprintf('%.4f',dosydata.viscositydisplay));
        set(hAnswer,'String','Press Calculate')
        set(hPassive,'String','Press Calculate')
        guidata(hMainFigure,dosydata);

    end

%	Method Callback
%   (chooses between SE or SEGWE)
    function method_Callback(~,~)
        %	Method Callback
        %   (The method choose in the popup menu will them select
        %   appropriate use of density and packing)
        dosydata=guidata(hMainFigure);
        item = get(hEstimationpopup,'Value'); % list entry
        list = get(hEstimationpopup,'String'); % list of solvents
        method = strtrim(list(item,:)); % retrieve solvent as string
        
        switch method
            case 'Stokes-Einstein-Gierer-Wirtz'
                set(hdensity,'Enable','off')
                set(hPacking,'Enable','off')
                dosydata.packing=1;
                dosydata.density=dosydata.peff/1000;
                set(hPacking,'String',sprintf('%.2f',dosydata.packing))
                set(hdensity,'String',sprintf('%.4f',dosydata.density))
                set(hCIButton,'Enable','on')
            case 'Stokes-Einstein'
                set(hdensity,'Enable','on')
                set(hPacking,'Enable','on')
                set(hCIButton,'Enable','off')
        end
        set(hAnswer,'String','Press Calculate')
        set(hPassive,'String','Press Calculate')
        guidata(hMainFigure,dosydata);
    end

%	Button Callback
%   (calculate and show results / output)
    function calculate_Callback(~,~)
        dosydata=guidata(hMainFigure);
        if dosydata.argument==0 || isnan(dosydata.argument)%No value to calculate
            errordlg('No value to Calculate!')
            set(hAnswer,'String','Error')
            set(hPassive,'String','Error')
            guidata(hMainFigure,dosydata);
            return
        else
            dosydata;
            format LONG
            % Constants
            kB = 1.380644e-23;
            NA = 6.02214e23;
            n=dosydata.viscositydisplay/1000;
            
            item = get(hEstimationpopup,'Value'); % list entry
            list = get(hEstimationpopup,'String'); % list of solvents
            method = strtrim(list(item,:)); % retrieve solvent as string
            
            switch method
                case 'Stokes-Einstein-Gierer-Wirtz'
                    %default
                    switch get(get(hEstimation_group,'SelectedObject'),'String')
                        case 'D to MW and rH'
                            Jc = ((dosydata.argument*(10^-10)*6*pi*n)/(kB*dosydata.temperature))*(((dosydata.MWs*3*(10^(-3)))/(4*pi*dosydata.peff*NA))^(1/3));
                            a2 = 3;
                            b2 = 3;
                            c2 = (2-2*Jc);
                            d2 = (-1)*(2*Jc);
                            p = (3*a2*c2-(b2^2))/(3*(a2^2));
                            q = (2*(b2^3)-(9*a2*b2*c2)+27*(a2^2)*d2)/(27*a2^3);
                            fac=4*(p^3)+27*(q^2);
                            if p<0 && fac>0
                                t = -2*(abs(q)/q)*((-p/3)^0.5)*(cosh((1/3)*acosh((-3/2)*(abs(q)/p)*((-3/p)^0.5))));
                            elseif p>0
                                t = -2*((p/3)^0.5)*(sinh((1/3)*asinh((3/2)*(q/p)*((3/p)^0.5))));
                            else
                                errordlg('Something went wrong')
                                set(hAnswer,'String','Error')
                                set(hPassive,'String','Error')
                            end
                            alpha = (t - (b2/(3*a2)));
                            dosydata.answer = dosydata.MWs/((alpha)^3);
                            radius=(((3*dosydata.answer*(10^(-3)))/(4*pi*dosydata.peff*NA))^(1/3))/(10^-12);
                            set(hAnswer,'String',sprintf('%.2f',dosydata.answer));
                            set(hPassive,'String',sprintf('%.2f',radius));
                        case 'MW to D and rH'
                            radius=(((dosydata.argument/(10^(3)))/((4*pi/3)*NA*(dosydata.peff)))^(1/3))/(10^-12);
                            alpha = ((dosydata.MWs/dosydata.argument)^(1/3));
                            dosydata.answer = ((kB*dosydata.temperature*(3*alpha/2 + 1/(1+alpha)))/(6*pi*n*(3*dosydata.argument*10^(-3)/(4*pi*dosydata.peff*NA))^(1/3)))/(10^-10);
                            set(hAnswer,'String',sprintf('%.2f',dosydata.answer));
                            set(hPassive,'String',sprintf('%.2f',radius));
                            
                        case 'rH to D and MW'
                            MW=(((dosydata.argument*(10^-12))^(3))*(4*pi/3)*NA*dosydata.peff)*(10^(3));
                            alpha = (dosydata.MWs/MW)^(1/3);
                            dosydata.answer = ((kB*dosydata.temperature*(3*alpha/2 + 1/(1+alpha)))/(6*pi*n*(dosydata.argument*(10^-12))))/(10^-10);
                            set(hAnswer,'String',sprintf('%.2f',dosydata.answer));
                            set(hPassive,'String',sprintf('%.2f',MW));
                    end
                    
                case 'Stokes-Einstein'
                    switch get(get(hEstimation_group,'SelectedObject'),'String')
                        case 'D to MW and rH'
                            if isnan(dosydata.packing) || isnan(dosydata.density)
                                errordlg('Stokes-Einstein requires packing factor and bulk density. Setting to default')
                                dosydata.packing=0.64;
                                dosydata.density=1.05;
                                set(hPacking,'String',num2str(dosydata.packing))
                                set(hdensity,'String',num2str(dosydata.density))
                                guidata(hMainFigure,dosydata);
                            else
                            end
                            dosydata.answer=((4*pi*dosydata.density*NA*(10^6))/(3*dosydata.packing))*((kB*dosydata.temperature)/((6*pi*n)*dosydata.argument*10^-10))^3;
                            radius=(((3*dosydata.packing*dosydata.answer)/(4*pi*dosydata.density*NA*(10^6)))^(1/3))/(10^-12);
                            passive=radius;
                            set(hAnswer,'String',sprintf('%.2f',dosydata.answer));
                            set(hPassive,'String',sprintf('%.2f',passive));
                        case 'MW to D and rH'
                            if isnan(dosydata.packing) || isnan(dosydata.density)
                                errordlg('Stokes-Einstein requires packing factor and bulk density. Setting to default')
                                dosydata.packing=0.64;
                                dosydata.density=1.05;
                                set(hPacking,'String',num2str(dosydata.packing))
                                set(hdensity,'String',num2str(dosydata.density))
                                guidata(hMainFigure,dosydata);
                            else
                            end
                            x1=((6*pi*n)/(kB*dosydata.temperature))*((3*dosydata.packing*dosydata.argument)/(4*pi*dosydata.density*(10^6)*NA))^(1/3);
                            dosydata.answer=(x1^-1)/(10^-10);
                            radius=(((3*dosydata.packing*dosydata.argument)/(4*pi*dosydata.density*NA*(10^6)))^(1/3))/(10^-12);
                            passive=radius;
                            set(hAnswer,'String',sprintf('%.2f',dosydata.answer));
                            set(hPassive,'String',sprintf('%.2f',passive));
                        case 'rH to D and MW'
                            if isnan(dosydata.packing) || isnan(dosydata.density)
                                errordlg('Stokes-Einstein requires packing factor and bulk density. Setting to default')
                                dosydata.packing=0.64;
                                dosydata.density=1.05;
                                set(hPacking,'String',num2str(dosydata.packing))
                                set(hdensity,'String',num2str(dosydata.density))
                                guidata(hMainFigure,dosydata);
                            else
                            end
                            dosydata.answer=((kB*dosydata.temperature)/(6*pi*n*(dosydata.argument*(10^-12))))/(10^-10);
                            MW=(((dosydata.argument*(10^-12))^(3))*(4*pi/3)*NA*dosydata.density/dosydata.packing)*(10^(6));
                            passive=MW;
                            set(hAnswer,'String',sprintf('%.2f',dosydata.answer));
                            set(hPassive,'String',sprintf('%.2f',passive));
                    end
            end
            format SHORT
            guidata(hMainFigure,dosydata);
        end
    end

%	Confidence Callback
%   (calculate and show results / confidence interval)
    function confidence_Callback(~,~)
        dosydata=guidata(hMainFigure);
        if dosydata.argument==0 || isnan(dosydata.argument)%No value to calculate
            errordlg('No value to Calculate!')
            set(hAnswer,'String','Error')
            set(hPassive,'String','Error')
            guidata(hMainFigure,dosydata);
            return
        else
            click_fig=figure('Units','normalized',...
                'MenuBar','none',...
                'Toolbar','none',...
                'NumberTitle','Off',...
                'Name','95% Confidence interval',...
                'WindowStyle','modal',...
                'OuterPosition',[0.5 0.5 0.15 0.4],...
                'Visible', 'Off');
            uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Method',...
                'FontWeight','bold',...
                'Position',[0.05 0.4 0.5 0.5 ],...
                'horizontalalignment','c');
            msg_met=uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Method',...
                'FontWeight','normal',...
                'Position',[0.63 0.4 0.25 0.5 ],...
                'horizontalalignment','c');
            uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Mode',...
                'FontWeight','bold',...
                'Position',[0.05 0.25 0.5 0.5 ],...
                'horizontalalignment','c');
            msg_mode=uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Mode',...
                'FontWeight','normal',...
                'Position',[0.61 0.25 0.3 0.5 ],...
                'horizontalalignment','c');
            msg_arg=uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Argument',...
                'FontWeight','bold',...
                'Position',[0.05 0.1 0.5 0.5 ],...
                'horizontalalignment','c');
            msg_pas=uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Passive',...
                'FontWeight','bold',...
                'Position',[0.05 -0.05 0.5 0.5 ],...
                'horizontalalignment','c');
            msg_ans=uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Answer',...
                'FontWeight','bold',...
                'Position',[0.05 -0.2 0.5 0.5 ],...
                'horizontalalignment','c');
            num_arg=uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Argument',...
                'FontWeight','normal',...
                'Position',[0.64 0.1 0.25 0.5 ],...
                'horizontalalignment','c');
            num_pas=uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Passive',...
                'FontWeight','normal',...
                'Position',[0.6 -0.05 0.35 0.5 ],...
                'horizontalalignment','c');
            num_ans=uicontrol(...
                'Parent',click_fig,...
                'Style','text',...
                'Units','Normalized',...
                'String','Answer',...
                'FontWeight','normal',...
                'Position',[0.6 -0.2 0.35 0.5 ],...
                'horizontalalignment','c');
            uicontrol(...
                'Parent',click_fig,...
                'Style','PushButton',...
                'String','Close',...
                'Units','normalized',...
                'Position',[0.35 0.05 0.3 0.1],...
                'Callback',{@clickokButton_Callback});
            
            format LONG
            % Constants
            sigma=1.96*0.1475; %RMS for 95% confidence in percentage
            kB = 1.380644e-23;
            NA = 6.02214e23;
            n=dosydata.viscositydisplay/1000;
            item = get(hEstimationpopup,'Value'); % list entry
            list = get(hEstimationpopup,'String'); % list of solvents
            method = strtrim(list(item,:)); % retrieve solvent as string
            
            switch method
                case 'Stokes-Einstein-Gierer-Wirtz'
                    set(msg_met,'String','SEGWE')
                    %default
                    switch get(get(hEstimation_group,'SelectedObject'),'String')
                        case 'D to MW and rH'
                            set(msg_mode,'String','D to MW and rH')
                            Jc = ((dosydata.argument*(10^-10)*6*pi*n)/(kB*dosydata.temperature))*(((dosydata.MWs*3*(10^(-3)))/(4*pi*dosydata.peff*NA))^(1/3));
                            a2 = 3;
                            b2 = 3;
                            c2 = (2-2*Jc);
                            d2 = (-1)*(2*Jc);
                            p = (3*a2*c2-(b2^2))/(3*(a2^2));
                            q = (2*(b2^3)-(9*a2*b2*c2)+27*(a2^2)*d2)/(27*a2^3);
                            fac=4*(p^3)+27*(q^2);
                            if p<0 && fac>0
                                t = -2*(abs(q)/q)*((-p/3)^0.5)*(cosh((1/3)*acosh((-3/2)*(abs(q)/p)*((-3/p)^0.5))));
                            elseif p>0
                                t = -2*((p/3)^0.5)*(sinh((1/3)*asinh((3/2)*(q/p)*((3/p)^0.5))));
                            else
                                errordlg('Something went wrong')
                                set(hAnswer,'String','Error')
                                set(hPassive,'String','Error')
                                return
                            end
                            alpha = (t - (b2/(3*a2)));
                            dosydata.answer = dosydata.MWs/((alpha)^3);
                            radius=(((3*dosydata.answer*(10^(-3)))/(4*pi*dosydata.peff*NA))^(1/3))/(10^-12);
                            passive=radius;
                            
                            % calculate 95% confidence interval
                            Jc = ((dosydata.argument*(1-sigma)*(10^-10)*6*pi*n)/(kB*dosydata.temperature))*(((dosydata.MWs*3*(10^(-3)))/(4*pi*dosydata.peff*NA))^(1/3));
                            a2 = 3;
                            b2 = 3;
                            c2 = (2-2*Jc);
                            d2 = (-1)*(2*Jc);
                            p = (3*a2*c2-(b2^2))/(3*(a2^2));
                            q = (2*(b2^3)-(9*a2*b2*c2)+27*(a2^2)*d2)/(27*a2^3);
                            fac=4*(p^3)+27*(q^2);
                            if p<0 && fac>0
                                t = -2*(abs(q)/q)*((-p/3)^0.5)*(cosh((1/3)*acosh((-3/2)*(abs(q)/p)*((-3/p)^0.5))));
                            elseif p>0
                                t = -2*((p/3)^0.5)*(sinh((1/3)*asinh((3/2)*(q/p)*((3/p)^0.5))));
                            else
                                errordlg('Something went wrong')
                                set(hAnswer,'String','Error')
                                set(hPassive,'String','Error')
                                return
                            end
                            alpha_plus = (t - (b2/(3*a2)));
                            answer_plus = dosydata.MWs/((alpha_plus)^3);
                            radius_plus=(((3*answer_plus*(10^(-3)))/(4*pi*dosydata.peff*NA))^(1/3))/(10^-12);
                            Jc = ((dosydata.argument*(1+sigma)*(10^-10)*6*pi*n)/(kB*dosydata.temperature))*(((dosydata.MWs*3*(10^(-3)))/(4*pi*dosydata.peff*NA))^(1/3));
                            a2 = 3;
                            b2 = 3;
                            c2 = (2-2*Jc);
                            d2 = (-1)*(2*Jc);
                            p = (3*a2*c2-(b2^2))/(3*(a2^2));
                            q = (2*(b2^3)-(9*a2*b2*c2)+27*(a2^2)*d2)/(27*a2^3);
                            fac=4*(p^3)+27*(q^2);
                            if p<0 && fac>0
                                t = -2*(abs(q)/q)*((-p/3)^0.5)*(cosh((1/3)*acosh((-3/2)*(abs(q)/p)*((-3/p)^0.5))));
                            elseif p>0
                                t = -2*((p/3)^0.5)*(sinh((1/3)*asinh((3/2)*(q/p)*((3/p)^0.5))));
                            else
                                errordlg('Something went wrong')
                                set(hAnswer,'String','Error')
                                set(hPassive,'String','Error')
                                return
                            end
                            alpha_minus = (t - (b2/(3*a2)));
                            answer_minus = dosydata.MWs/((alpha_minus)^3);
                            radius_minus=(((3*answer_minus*(10^(-3)))/(4*pi*dosydata.peff*NA))^(1/3))/(10^-12);
                            
                            set(hAnswer,'String',sprintf('%.2f',dosydata.answer));
                            set(hPassive,'String',sprintf('%.2f',passive));
                            
                            set(msg_ans,'String','Molecular Weight / (g mol-1)')
                            set(msg_arg,'String','Diffusion Coefficient / (10^-10 m2 s-1)')
                            set(msg_pas,'String','Hydrodynamic radius / (10^-12 m)')
                            set(num_arg,'String',sprintf('%.2f',dosydata.argument))
                            set(num_ans,'String',[sprintf('%.2f',answer_minus) ' - ' sprintf('%.2f',answer_plus)])
                            set(num_pas,'String',[sprintf('%.2f',radius_minus) ' - ' sprintf('%.2f',radius_plus)])
                            
                        case 'MW to D and rH'
                            set(msg_mode,'String','MW to D and rH')
                            radius=(((dosydata.argument/(10^(3)))/((4*pi/3)*NA*dosydata.peff))^(1/3))/(10^-12);
                            alpha = ((dosydata.MWs/dosydata.argument)^(1/3));
                            dosydata.answer = ((kB*dosydata.temperature*(3*alpha/2 + 1/(1+alpha)))/(6*pi*n*(3*dosydata.argument*10^(-3)/(4*pi*dosydata.peff*NA))^(1/3)))/(10^-10);
                            passive=radius;
                            
                            % calculate 95% confidence interval
                            radius_plus=radius*(1-sigma);
                            answer_plus=dosydata.answer*(1-sigma);
                            radius_minus=radius*(1+sigma);
                            answer_minus=dosydata.answer*(1+sigma);
                            
                            set(hAnswer,'String',sprintf('%.2f',dosydata.answer));
                            set(hPassive,'String',sprintf('%.2f',passive));
                            
                            set(msg_arg,'String','Molecular Weight / (g mol-1)')
                            set(msg_ans,'String','Diffusion Coefficient / (10^-10 m2 s-1)')
                            set(msg_pas,'String','Hydrodynamic radius / (10^-12 m)')
                            set(num_arg,'String',sprintf('%.2f',dosydata.argument))
                            set(num_ans,'String',[sprintf('%.2f',answer_plus) ' - ' sprintf('%.2f',answer_minus)])
                            set(num_pas,'String',[sprintf('%.2f',radius_plus) ' - ' sprintf('%.2f',radius_minus)])
                            
                        case 'rH to D and MW'
                            set(msg_mode,'String','rH to D and MW')
                            MW=(((dosydata.argument*(10^-12))^(3))*(4*pi/3)*NA*dosydata.peff)*(10^(3));
                            alpha = (dosydata.MWs/MW)^(1/3);
                            dosydata.answer = ((kB*dosydata.temperature*(3*alpha/2 + 1/(1+alpha)))/(6*pi*n*(dosydata.argument*(10^-12))))/(10^-10);
                            passive=MW;
                            
                            % calculate 95% confidence interval
                            MW_minus=MW*(1-sigma);
                            answer_plus=dosydata.answer*(1-sigma);
                            MW_plus=MW*(1+sigma);
                            answer_minus=dosydata.answer*(1+sigma);
                            
                            set(hAnswer,'String',sprintf('%.2f',dosydata.answer));
                            set(hPassive,'String',sprintf('%.2f',passive));
                            
                            set(msg_pas,'String','Molecular Weight / (g mol-1)')
                            set(msg_ans,'String','Diffusion Coefficient / (10^-10 m2 s-1)')
                            set(msg_arg,'String','Hydrodynamic radius / (10^-12 m)')
                            set(num_arg,'String',sprintf('%.2f',dosydata.argument))
                            set(num_ans,'String',[sprintf('%.2f',answer_plus) ' - ' sprintf('%.2f',answer_minus)])
                            set(num_pas,'String',[sprintf('%.2f',MW_minus) ' - ' sprintf('%.2f',MW_plus)])
                    end
                    
            end
            format SHORT
        end
        guidata(hMainFigure,dosydata);
        
        set(click_fig,'Visible','On')
        function clickokButton_Callback(~,~)
            delete(click_fig)
        end
    end

%   Show feedback information
if info == 1
uiwait(msgbox({'Please send us feedback at' 'dalpgt@unicamp.br'},...
    '','help','modal'));
else
end 
%% --- Adding extra solvents (optional)
%   Including new solvents is possible by following the next steps:
%   1) Add solvent name to solvent popup menu option (line 161)
%   2) Create solvent parameters "inside solvent_Callback" (line 543)
%      These must include 'a' , 'b' , 'peff' and 'MWs'.
%      An example is set at line 746.
end
%   Guilherme Dal Poggetto, PhD
%   Physical Organic Chemistry Laboratory (LFQO)
%   Institute of Chemistry, UNICAMP
%   Campinas, SP - Brazil
%   guilherme.dalpoggetto@manchester.ac.uk
%   mathias.nilsson@manchester.ac.uk