function [rwdata,brukerdata,dispaxis]=spen2GNAT_psyche(FilePath,NBC,refprof_y_n,brukerdata,acqref,chirpref,add_sm_proc)
%**********************************************************************************************************
% [rwdata,brukerdata,dispaxis]=spen2GNAT(FilePath,NBC,refprof_y_n,brukerdata,acqref,chirpref,add_sm_proc)
% Converts SPEN DOSY data to GNAT compatible format
%  - also used for processing 1D and 2D acquistion, chirp reference profile
%       -Input parameters
%              - FilePath - the path of the file
%              - NBC - number for baseline correction
%              - refprof_y_n - file needed to be processed is
%                                   acquisition reference profile (0),
%                                  chirp+acquisition reference profile (1),
%                                   SPEN DOSY data (2)
%              - brukerdata - intial values are obained form
%                             brukerimport_SPEN function
%        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%          remaining inputs are used only for SPEN DOSY data processing
%              - acqref - 'yes' or 'no' for acquisition reference profile
%                           correction
%              - chirpref - 'on' or 'off' for chirp+acquisition reference
%                            profile correction
%              - add_sm_proc - 'yes' or 'no'; used to get rid of unwanted
%                            spikes outside the region of interest,
%                            obatined after inverse fourier transformation.
%       -Output parameters
%              - rwdata - Fourier transformed data; used for acquisiton and
%                         chirp+acquistion profiles not required for SPEN
%                         DOSY data
%              - brukerdata - contains many parameters, rwdata and fid
%                           compatible to GNAT are few of them
%              - dispaxis - display axis (in meters); used for
%                           interpolation of refrence profiles; this output
%                           is not required for SPEN DOSY data
%**********************************************************************************************************

% reads spendosy data and save the relevent parameters to brukerdata structure.
[brukerdata]=readspendosy_GNAT_psyche(FilePath,brukerdata);

% After reading the data it should be decided whether the data dimension is
% 1 or 2. This portion of the code is mainly used to process 1D reference
% profiles (used only if parameter refprof_y_n=0 or 1 in spen2GNAT input 
% argument)
if brukerdata.DataDim==1
    complexdata_full = readfid([FilePath '/fid'],brukerdata.byte_format,brukerdata.byte_size);
    % remove trailing zeros
    complexdata_full(brukerdata.NP+1:end) = [];
    
    % Remove the offset
    complexdata_full=complexdata_full-mean(complexdata_full(1:round(brukerdata.NP/ NBC)));
    
    [complexdata,~]=apodization_k_t(complexdata_full,1,brukerdata.apdfunc,brukerdata);
    
    % zero filling in the ultrafast dimension
    nZFK=max(brukerdata.nZFK,brukerdata.NP);
    brukerdata.nZFK=nZFK;
    kdatazf = zeros(nZFK,1);
    kdatazf(1+(nZFK-brukerdata.NP)/2:brukerdata.NP+(nZFK-brukerdata.NP)/2) = complexdata;
    kdata = kdatazf;
    
    vector_deltat =  (-nZFK/2:nZFK/2-1) * brukerdata.dw;
    dela_nu = (brukerdata.acqus.O1 -  brukerdata.expO1);
    exp_factor_acq = 1* vector_deltat' * dela_nu * 1i * 2*pi;
    kdata = kdata.* exp(exp_factor_acq);
    
    % Fourier tranform
    rdata = fftshift(fft(fftshift(kdata)));
    
    % Magnitude mode
    rdata = abs(rdata);
    dispaxis = linspace(0.5,-0.5,nZFK)'*brukerdata.SW*2*pi/...
        (brukerdata.gamma1H*brukerdata.Gdmax*brukerdata.Gacq/100); % FOV (mm)
    
    % Reference profile
    %     procdata = rdata;
    rwdata = rdata;
    brukerdata.dispaxis = dispaxis;
    
% if the data is of the 2D form. This portion of the code is used to 
% process 2D reference profiles (if refprof_y_n=0 or 1) 
% or SPEN DOSY data (if refprof_y_n=2)  
else
    brukerdata.gammaB1=1/(brukerdata.P1*4)*(brukerdata.SP1/brukerdata.PL1)^(1/2);% pulse strength
    complexdata_full = readfid([FilePath '/ser'],brukerdata.byte_format,brukerdata.byte_size);
    switch brukerdata.pureshift
        case 0
            complexdata=complexdata_full((brukerdata.numscan-1)*brukerdata.td/2+1:(brukerdata.numscan)*brukerdata.td/2);
        case 1
            complexdata=complexdata_full;
    end
    if brukerdata.acqus.GRPDLY>1
        % Remove the group delay
        nD=brukerdata.acqus.GRPDLY;
        chop=floor(nD) ;
        complexdata = circshift(complexdata,-chop);
        brukerdata.p= (nD-chop);        
    end
    
    % Reshape the data
    NT = 2*brukerdata.N2*brukerdata.Ni;
    brukerdata.NT=NT;
    NK = brukerdata.NP/(2*brukerdata.N2);
    brukerdata.NK=NK;
    nZFK=max(brukerdata.nZFK,NK);
    brukerdata.nZFK=nZFK;
    switch brukerdata.pureshift
        case 0
            ktdata = reshape(complexdata,NK,NT);
            
            % % Remove the offset
            DC_offset1=ktdata(1:round(brukerdata.NK/NBC),:);
            DC_offset1=mean(DC_offset1,1);
            DC_offset1=mean(DC_offset1,2);
            ktdata=ktdata-DC_offset1;
            
            [ktdata,~]=apodization_k_t(ktdata,1,brukerdata.apdfunc,brukerdata);
            
            % zero filling in the ultrafast dimension
            ktdatazf = zeros(nZFK,NT);
            ktdatazf(1+(nZFK-NK)/2:NK+(nZFK-NK)/2,:) = ktdata;
            
            % Process only the odd echoes
            ktdataoddseq = ktdatazf(:,1:2:2*brukerdata.N2*brukerdata.Ni-1);
            
            
            if brukerdata.Ni == 1
                ktdataodd = ktdataoddseq;
            else
                
                % Interleave the data
                ktdataodd = zeros(size(ktdataoddseq));
                for k = 1:brukerdata.N2
                    for l = 1:brukerdata.Ni
                        ktdataodd(:,(k-1)*brukerdata.Ni+l) = ktdataoddseq(:,k+brukerdata.N2*(l-1));
                    end
                end
            end
        case 1
            complexdataarray=reshape(complexdata,length(complexdata)/brukerdata.NPS,brukerdata.NPS);
            complexdataarray(brukerdata.NP+1:end,:)=[];
            clear complexdata
            
            
            complexdata=complexdataarray(:);
            
            ktpdata = reshape(complexdata,brukerdata.NK,brukerdata.NT,brukerdata.NPS);
            
            for l = 1:brukerdata.NPS
                ktpdata(:,:,l)=circshift(ktpdata(:,:,l),1*l,1);
            end
            
            ktpdataodd = ktpdata(:,1:2:end,:);
            ktdataodd_test = zeros(brukerdata.NK,brukerdata.npChunk*brukerdata.NPS);
            
            for i=1:brukerdata.NPS
                ktdataodd_test(:,brukerdata.npChunk*(i-1)+1:brukerdata.npChunk*i) = ktpdataodd(:,(1:brukerdata.npChunk),i);
            end
            
            % apodization in ultrafast dimension
            [ktdataodd_test,~] = apodization_k_t(ktdataodd_test,1,brukerdata.apdfunc,brukerdata);
            % zero filling in the ultrafast dimension
            ktdatazf = zeros(nZFK,brukerdata.npChunk*brukerdata.NPS);
            ktdatazf(1+(nZFK-NK)/2:NK+(nZFK-NK)/2,:) = ktdataodd_test;
            ktdataodd=ktdatazf;
    end
    % Shearing step to correct unbalanced acquisition gradients
    if  exist ([FilePath '/shearval.txt'],'file') == 0 %exist('shearvalue.mat') == 0
        [ktdataodd,shearvalue] = shear_gui(ktdataodd);
        %save('shearvalue')
        shearfile = fopen([FilePath '/shearval.txt'],'w');
        fprintf(shearfile, '%f', shearvalue);
        fclose(shearfile);
    else
        shearquest = questdlg('Shearing has already been done. Do you want to use the already existing file ?',...
            'Shearing file', 'Yes', 'No', 'Yes');
        switch shearquest
            case 'Yes'
                %load('shearvalue')
                shearfile = fopen([FilePath '/shearval.txt'],'r');
                shearvalue = fscanf(shearfile, '%f');
                fclose(shearfile);
                linshear = linspace(0,shearvalue,size(ktdataodd,2));
                for i=1:size(ktdataodd,2)
                    ktdataodd(:,i) = CJshear(ktdataodd(:,i),0,linshear(i));
                end
            case 'No'
                [ktdataodd,shearvalue] = shear_gui(ktdataodd);
                shearfile = fopen([FilePath '/shearval.txt'],'w');
                fprintf(shearfile, '%f', shearvalue);
                fclose(shearfile);
        end
    end
    [ktdataodd,~] = apodization_k_t(ktdataodd,0,brukerdata.apdfunc_t,brukerdata);
    
    % Zero fill in the conventional dimension
    nZFT = max(brukerdata.nZFT,size(ktdataodd,2));
    brukerdata.nZFT = nZFT;
    ktdataoddzf = zeros(nZFK,nZFT);
    ktdataoddzf(:,1:size(ktdataodd,2)) = ktdataodd;
    ktdataodd = ktdataoddzf;
    
    % Axis for data plotting
    SWconv=1/(2*brukerdata.Ta)/brukerdata.B0;
    brukerdata.SWconv=SWconv;
    ppm_axis=linspace(0.5,-0.5,nZFT)'*SWconv+brukerdata.offsetconv; %(ppm)
    
    
    % Axis for data plotting:
    dispaxis = linspace(0.5,-0.5,nZFK)'*brukerdata.SW*2*pi/...
        (brukerdata.gamma1H*brukerdata.Gdmax*brukerdata.Gacq/100); % FOV (mm)
    
    
    switch brukerdata.phmode
        case 'mc'
            kwdata = fftshift(fft(ktdataodd,[],2),2);
            rwdata_wocorr  = fftshift(fft(fftshift(kwdata,1),[],1),1);
        case 'pk'
            brukerdata.FilePath=FilePath;
            kwdata = fftshift(fft(ktdataodd,[],2),2);
            rwdata  = fftshift(fft(fftshift(kwdata,1),[],1),1);
            [rwdataph]=uf_ph_man_GNAT(rwdata,brukerdata);
            rwdata = real(rwdataph);
            kwdata = fftshift(ifft(fftshift(rwdata,1),[],1),1);
    end
    
    
    
    vector_deltat =  (-nZFK/2:nZFK/2-1) * brukerdata.dw;
    dela_nu = (ppm_axis - brukerdata.offsetconv)* brukerdata.B0;
    % exponential function to be multiplied for removing the effect of
    % offset on sample position during acquisition
    exp_factor_acq = 1* vector_deltat' * dela_nu' * 1i * 2*pi;
    
    % % k-t space tilt correction during Chirp
    exp_factor_chirp = 1* vector_deltat' * (brukerdata.Gdmax*brukerdata.Gacq/100) *...
        dela_nu' * 2*1i*pi / (brukerdata.Gdmax*brukerdata.GPZ0);
    kwdata_tiltacq_corrected = kwdata .* exp(exp_factor_acq);
    
    % FFT of acquisition tilt-corrected data to obtain acquisition reference profile
    if refprof_y_n==0
        rwdata_acq_corrected = fftshift(fft(fftshift(kwdata_tiltacq_corrected,1),[],1),1);
        rwdata_acq_corrected = abs(rwdata_acq_corrected);
        [~,C] = find(rwdata_acq_corrected == max(max(rwdata_acq_corrected)));
        rwdata = abs(rwdata_acq_corrected(:,C));
    % chirp tilt-correction and FFT of (chirp+acquisition) tilt-corrected
    % data to obtain (chirp+acquisition) reference profile
    elseif refprof_y_n==1
        kwdata_chirp_acq_corrected = kwdata_tiltacq_corrected .* exp(-exp_factor_chirp);
        rwdata_chirp_acq_corrected = fftshift(fft(fftshift(kwdata_chirp_acq_corrected,1),[],1),1);
        [~,C] = find(rwdata_chirp_acq_corrected == max(max(rwdata_chirp_acq_corrected)));
        rwdata = abs(rwdata_chirp_acq_corrected(:,C));
     % to process SPEN DOSY data with/without chirp and acquisition refprof
     % correction
    elseif refprof_y_n==2
        rwdata_tiltacq_corr  = fftshift(fft(fftshift(kwdata_tiltacq_corrected,1),[],1),1);
        switch acqref
            % to obain acquisition referece profile and correcting the data
            case 'on'
                if brukerdata.pureshift==1
                    temp=1;
                    brukerdata.pureshift=0;
                else
                    temp=0;
                end
                [brukerdata.refexpno_epsi,~,brukerdata.dispaxis_epsi]=...
                    spen2GNAT_psyche(brukerdata.FilePath_epsi,brukerdata.NBC_acqref,0,brukerdata);
                if brukerdata.dispaxis_epsi(1)<dispaxis(1)
                    brukerdata.refexpno_epsi = ...
                          interp1(brukerdata.dispaxis_epsi,brukerdata.refexpno_epsi,(dispaxis),...
                                                                                'linear','extrap');
                else
                    brukerdata.refexpno_epsi = ...
                              interp1(brukerdata.dispaxis_epsi,brukerdata.refexpno_epsi,(dispaxis));
                end
%                 brukerdata.refexpno_epsi = interp1(brukerdata.dispaxis_epsi,brukerdata.refexpno_epsi,(dispaxis));
                rwdata_tiltacq_acqref_corr = rwdata_tiltacq_corr ./ brukerdata.refexpno_epsi;
                if temp==1
                    brukerdata.pureshift=1;
                else
                   brukerdata.pureshift=0; 
                end
                % after dividing the acq tilt corrected data by acquistion
                % reference profile unwanted spikes arises. To remove those
                % spikes an additional smooting is performed
                switch add_sm_proc
                    case 'yes'
                        range = find(dispaxis<(brukerdata.fitLengthPos) & dispaxis>(brukerdata.fitLengthNeg));
                        if rem(size(range,1),2) == 0
                            np_smooth=floor((nZFK-size(range,1))/2);
                            np_nonsmooth=size(range,1);
                        else
                            np_smooth=floor((nZFK-size(range,1)+1)/2);
                            np_nonsmooth=size(range,1)-1;
                        end
                        amp_mid=ones(1,np_nonsmooth);
                        smoothed_fraction_l=(sin(linspace(0,pi/2,np_smooth))).^128;
                        smoothed_fraction_r=fliplr(smoothed_fraction_l);
                        amplitudes=horzcat(smoothed_fraction_l,amp_mid,smoothed_fraction_r);
                        amplitudes = circshift(amplitudes,4);
                        rwdata_tiltacq_acqref_corr_w_sm = rwdata_tiltacq_acqref_corr .* repmat(amplitudes',1,nZFT);
                        rwdata_tiltacq_acqref_corr = rwdata_tiltacq_acqref_corr_w_sm;
                    case 'no'
                        rwdata_tiltacq_acqref_corr_wo_sm = rwdata_tiltacq_acqref_corr;
                        rwdata_tiltacq_acqref_corr = rwdata_tiltacq_acqref_corr_wo_sm;
                end
                kwdata_tiltacq_acqref_corr = fftshift(ifft(fftshift(rwdata_tiltacq_acqref_corr,1),[],1),1);
                kwdata_tiltacqchirp_acqref_corr = kwdata_tiltacq_acqref_corr .* exp(-exp_factor_chirp);
                rwdata_tiltacqchirp_acqref_corr = fftshift(fft(fftshift(kwdata_tiltacqchirp_acqref_corr,1),[],1),1);
                switch chirpref
                % to obain chirp reference profile and correcting the data
                    case 'on'
                        if brukerdata.pureshift==1
                            temp=1;
                            brukerdata.pureshift=0;
                        else
                            temp=0;
                        end
                        [brukerdata.refexpno_chirp_epsi,~,brukerdata.dispaxis_chirp]=...
                            spen2GNAT_psyche(brukerdata.FilePath_chirp_epsi,brukerdata.NBC_chirpacqref,1,brukerdata);
                        if brukerdata.dispaxis_epsi(1)<dispaxis(1)
                            brukerdata.refexpno_chirp_epsi = ...
                                interp1(brukerdata.dispaxis_chirp,brukerdata.refexpno_chirp_epsi,(dispaxis),...
                                                                                                  'linear','extrap');
                        else
                            brukerdata.refexpno_chirp_epsi = ...
                            interp1(brukerdata.dispaxis_chirp,brukerdata.refexpno_chirp_epsi,(dispaxis));
                        end
%                         brukerdata.refexpno_chirp_epsi = ...
%                             interp1(brukerdata.dispaxis_chirp,brukerdata.refexpno_chirp_epsi,(dispaxis));
                        corr_factor = brukerdata.refexpno_chirp_epsi ./ brukerdata.refexpno_epsi;
                        rwdata_tiltacqchirp_acqchirpref_corrected = rwdata_tiltacqchirp_acqref_corr ./ corr_factor;
                        rwdata=rwdata_tiltacqchirp_acqchirpref_corrected;
                        if temp==1
                            brukerdata.pureshift=1;
                        else
                            brukerdata.pureshift=0;
                        end
                    case 'off'
                        rwdata=rwdata_tiltacqchirp_acqref_corr;
                end
            % to perform only the chirp profile correction without 
            % acquisition profile correction 
            case 'off'
                kwdata_tiltacq_corr = fftshift(ifft(fftshift(rwdata_tiltacq_corr,1),[],1),1);
                kwdata_tiltacqchirp_corr = kwdata_tiltacq_corr .* exp(-exp_factor_chirp);
                rwdata_tiltacqchirp_corr = fftshift(fft(fftshift(kwdata_tiltacqchirp_corr,1),[],1),1);
                switch chirpref
                    case 'on'
                        if brukerdata.pureshift==1
                            temp=1;
                            brukerdata.pureshift=0;
                        else
                            temp=0;
                        end
                        [brukerdata.refexpno_epsi,~,brukerdata.dispaxis_epsi]=...
                            spen2GNAT_psyche(brukerdata.FilePath_epsi,brukerdata.NBC_acqref,0,brukerdata);
                        if brukerdata.dispaxis_epsi(1)<dispaxis(1)
                            brukerdata.refexpno_epsi = ...
                                interp1(brukerdata.dispaxis_epsi,brukerdata.refexpno_epsi,(dispaxis),...
                                'linear','extrap');
                        else
                            brukerdata.refexpno_epsi = ...
                                interp1(brukerdata.dispaxis_epsi,brukerdata.refexpno_epsi,(dispaxis));
                        end
%                         brukerdata.refexpno_epsi = interp1(brukerdata.dispaxis_epsi,brukerdata.refexpno_epsi,(dispaxis));
                        [brukerdata.refexpno_chirp_epsi,~,brukerdata.dispaxis_chirp]=...
                            spen2GNAT_psyche(brukerdata.FilePath_chirp_epsi,brukerdata.NBC_chirpacqref,1,brukerdata);
                        if brukerdata.dispaxis_epsi(1)<dispaxis(1)
                            brukerdata.refexpno_chirp_epsi = ...
                                interp1(brukerdata.dispaxis_chirp,brukerdata.refexpno_chirp_epsi,(dispaxis),...
                                                                                                  'linear','extrap');
                        else
                            brukerdata.refexpno_chirp_epsi = ...
                            interp1(brukerdata.dispaxis_chirp,brukerdata.refexpno_chirp_epsi,(dispaxis));
                        end
                        corr_factor = brukerdata.refexpno_chirp_epsi ./ brukerdata.refexpno_epsi;
                        rwdata_tiltacqchirp_chirpref_corrected = rwdata_tiltacqchirp_corr ./ corr_factor;
                        rwdata=(rwdata_tiltacqchirp_chirpref_corrected);
                        if temp==1
                            brukerdata.pureshift=1;
                        else
                            brukerdata.pureshift=0;
                        end
                    case 'off'
                        rwdata=rwdata_tiltacqchirp_corr;
                end
        end
        dispaxis=flipud(dispaxis);
        brukerdata.dispaxis = dispaxis;
        brukerdata.ppm_axis = flipud(ppm_axis);
        
        
        % Find artefact location
        artefact_pos=abs(ppm_axis-brukerdata.offsetconv);
        artefact_ind=find(min((artefact_pos))==artefact_pos);
        artefact_range=artefact_ind(1)-10-(nZFT/brukerdata.N2/brukerdata.Ni):artefact_ind(1)+10+(nZFT/brukerdata.N2/brukerdata.Ni);
        brukerdata.artefact_range=artefact_range;
        
        % selection of region of interest
        samplelength=find(dispaxis<(brukerdata.fitLengthPos)...
                                    & dispaxis>(brukerdata.fitLengthNeg));  
        zi=samplelength(1);
        zf=samplelength(end);
        
        % for waterfall plot
        if brukerdata.wfplot==1
            ppm_axis_2d=repmat(ppm_axis,1,size(dispaxis,1));
            dispaxis_2d=repmat(1000*dispaxis.',size(ppm_axis,1),1);
%             rwdata=(rwdata);
%             rwdata = abs(rot90(rwdata.',2));
            figure('Name',brukerdata.wfplot_fig,'NumberTitle','off');
            plot3(ppm_axis_2d,fliplr(dispaxis_2d),abs(rot90(rwdata.',2)),'b');
            set(gca,'Xdir','reverse','Ydir','reverse')
            xlim(brukerdata.wfplot_xlim); ylim(brukerdata.wfplot_ylim)
            xlabel('ppm'); ylabel('Z(mm)')
            %             ax=gca; ax.View=[1.578000000000000e+02 22.98];
            ax.FontSize=12;
            ax.ZAxis.Visible='off';
            ax.FontWeight='Bold';
            %             ax.XLabel.Position=[2.866357371552127 7.507007045172827 -34.69507604258888];
            %             ax.YLabel.Position=[3.877080523542805 0.221887052403929 -28.299716078279744];
            if brukerdata.savefig==1
                saveas(gcf,brukerdata.wfplot_path);
                currentFolder=pwd;
                cd (brukerdata.wfplot_pathsvg)
                print(gcf,brukerdata.wfplot_fig,'-dsvg','-r600')
                cd (currentFolder)
                close (brukerdata.wfplot_fig)
                % gyan ke liye
                % figure;hold on;for k=51:80;plot(abs(rwdata(k,300:750)),'color',mymap(k-50,:));end;
            else
                currentFolder=pwd;
                cd (currentFolder)
                close (brukerdata.wfplot_fig)
            end
        end
        
        % Fourier transformed data in the region of interest
        rwdata=fliplr(rwdata(zi:zf,:));
        rwdata=rwdata./max(rwdata(:));
        switch brukerdata.phmode
        case 'mc'
        rwdata = abs(rot90(rwdata.',2));
        case 'pk'
            rwdata = real(rot90(rwdata.',2));
        end
%         if brukerdata.islinearchirp==1
%             rwdata=fliplr(rwdata);
%         elseif brukerdata.islinearchirp==0
%             rwdata=(rwdata);
%         end
        % Saving the data for GNAT 
        brukerdata.rwdata=rwdata;
        brukerdata.np=size(rwdata,1);
        brukerdata.arraydim=size(rwdata,2);        
        FID_rtdata=ifft(flipud((fftshift(rwdata,1))),[],1);
        FID_rtdata(1,:)=FID_rtdata(1,:).*2;
        brukerdata.FID=FID_rtdata;
        brukerdata.Ge= brukerdata.Gdmax*brukerdata.GPZ0;
        brukerdata.zaxis=dispaxis(zi:zf);
        brukerdata.ngrad=length(samplelength);
        
        switch brukerdata.nonuniform_grad_corr
            case 'yes'
                brukerdata.Ge_corrected=brukerdata.GPZ0*polyval(brukerdata.grad_coeff,1000*brukerdata.zaxis);
            case 'no'
                brukerdata.Ge_corrected=brukerdata.Ge.*ones(size(brukerdata.zaxis,1),1);
        end
        % Defining the values of effective gradient according to linear
        % chirp and quadratic spacing chirp pulse
        if brukerdata.islinearchirp==1
            %             Geff=2*brukerdata.gamma*brukerdata.Ge^2/(brukerdata.BW*2*pi)...
            %                 *brukerdata.zaxis+brukerdata.Ge;
            if brukerdata.pureshift==0
                Geff=brukerdata.Ge_corrected.*(2*brukerdata.gamma*brukerdata.Ge/...
                    (brukerdata.BW*2*pi).*brukerdata.zaxis+1);
                brukerdata.Gzlvl=fliplr(Geff');
            else
                alpha=0.5-brukerdata.gamma*brukerdata.Ge*...
                                             brukerdata.zaxis/(2*pi*brukerdata.BW);
%                 fr=(brukerdata.D30-(2-alpha)*brukerdata.Te)/brukerdata.Te;
%                 Geff=brukerdata.Ge_corrected.*sqrt((3+fr).*alpha.^2-(7/3)*alpha.^3);
                Geff=brukerdata.Ge_corrected.*alpha.*sqrt((brukerdata.D30+...
                                  brukerdata.Te -(4/3)*alpha*brukerdata.Te)...
                                                         ./brukerdata.D30);
                brukerdata.alpha=alpha;
%                 brukerdata.Gzlvl=fliplr(Geff');
                brukerdata.Gzlvl=(Geff');
            end
        elseif brukerdata.islinearchirp==0
            %             Geff=2*brukerdata.Ge*sqrt(-brukerdata.gamma*brukerdata.Ge...
            %                 *brukerdata.zaxis/(2*pi*brukerdata.BW)+0.5);
            Geff=2*brukerdata.Ge_corrected.*sqrt(-brukerdata.gamma*brukerdata.Ge...
                *brukerdata.zaxis/(2*pi*brukerdata.BW)+0.5);
            brukerdata.Gzlvl=fliplr(Geff');
        end
    end
end