% Performs free induction decay apodization. Supports 1D, 2D free induction
% decays
%
%                     fid=apodization(fid,k,window_type,inputp)
%
% Arguments:
%
%     fid         - the free induction decay. The function expects a column
%                   vector in the case of 1D FID, a 2D matrix with the time
%                   origin located at the (1,1) corner point in the case of
%                   a 2D FID, and a 3D matrix with the time origin located
%                   at the (1,1,1) corner point in the case of a 3D FID.
%       k         - refers to apodization dimension. If k is 0, apodization  
%                   is performed in time 't' diemsion and if k is 1, 
%                   apodization is performed in inverse z-space 'k' dim.
%
%     window_type - type of the window function. The following window func-
%                   tion types are supported: 
% 
%                   for t-dimension
%
%                   'sin' - use sinusoidal function 
%
%                   'cos' - use cosinusoidal function
%
%                   'exp' - use exponential function
%
%                   'gaussian' - use gaussian function
%
%                   'sqsin' - use square of sinusoidal function 
%
%                   'sqcos' - use square of cosinusoidal function
% 
%                   for k-dimension
% 
%                   'Hamming and Hann' - Hamming and Hann function
%                     
%                   'Blackman' - Blackman function
%                   
%                   'Nuttall'  - Nuttall function
%           
%                   'Blackman-Nuttall' - Blackman-Nuttall function
%                   
%                   'Blackman-Harris'  - Blackman-Harris function
%
%     inputp     - input parameters for apodization such as zero filling,
%                  apodization time constant, alpha for k-dim apodization 
%                  and so on
% Outputs:
%
%     fid   - apodised free induction decay
%
% Note: if an array of 1D FIDs is supplied, the FIDs must be in columns.
%
% i.kuprov@soton.ac.uk
% modified by Rituraj to add k-dimension apodization 
%
% <http://spindynamics.org/wiki/index.php?title=apodization.m>

function [fid,apdfunc]=apodization_k_t(fid,k,window_type,inputp)

if k==0
    procconvaxis = linspace(0,1,size(fid,2));
    switch window_type
    case 'sin'
        apdfunc = sin(pi*procconvaxis);%,inputp.nZFK,1);
        fid = fid .* apdfunc; 
    case 'cos'
        apdfunc = cos(pi/2*procconvaxis);%,inputp.nZFK,1);
        fid = fid .* apdfunc; 
    case 'exp'
        apdfunc = exp(-inputp.apodtime*procconvaxis);%,inputp.nZFK,1);
        fid = fid .* apdfunc; 
    case 'gaussian'                                                                        
        apdfunc = exp(-inputp.apodtime*procconvaxis.^2);%,inputp.nZFK,1);
        fid = fid .* apdfunc; 
    case 'sqsin'                                                                         
        apdfunc = sin(pi*procconvaxis).^2;%,inputp.nZFK,1);
        fid = fid .* apdfunc; 
    case 'sqcos'
        apdfunc = cos(pi*procconvaxis).^2;%,inputp.nZFK,1);
        fid = fid .* apdfunc; 
    end
elseif k==1
%     procufaxis = linspace(0,1,inputp.NK)';
%     procufaxis=reshape(procufaxis,inputp.NK,1);
      procufaxis = linspace(0,1,size(fid,1))';
      procufaxis=reshape(procufaxis,size(fid,1),1);
    switch window_type
    case 'Hamming and Hann'
         apdfunc = (inputp.alpha-(1- inputp.alpha)*cos(2*pi*procufaxis));%,1,inputp.NT);
         fid = fid .* apdfunc; 
    case 'Blackman'
         apdfunc = (7938/18608-(9240/18608)...
                           *cos(2*pi*procufaxis)...
                                 +(1430/18608)*cos(4*pi*procufaxis));%,1,inputp.NT);
         fid = fid .* apdfunc; 
    case 'Nuttall'
         apdfunc = (0.355768-0.487396*cos(2*pi*procufaxis)...
                                 +0.144232*cos(4*pi*procufaxis)...
                                            -0.012604*cos(6*pi*procufaxis));%,1,inputp.NT);
         fid = fid .* apdfunc; 
    case 'Blackman-Nuttall'                                                                        
         apdfunc = (0.3635819-0.4891775*cos(2*pi*procufaxis)...
                                +0.1365995*cos(4*pi*procufaxis)...
                                        -0.0106411*cos(6*pi*procufaxis));%,1,inputp.NT);
         fid = fid .* apdfunc; 
    case 'Blackman-Harris'                                                                         
         apdfunc = (0.35875-0.48829*cos(2*pi*procufaxis)...
                                +0.14128*cos(4*pi*procufaxis)...
                                            -0.01168*cos(6*pi*procufaxis));%,1,inputp.NT);
         fid = fid .* apdfunc; 
    case 'Fla-top'
         apdfunc = (0.21557895-0.41663158*cos(2*pi*procufaxis)...
                        +0.277263158*cos(4*pi*procufaxis)...
                                 -0.083578947*cos(6*pi*procufaxis)...
                                        +0.006947368*cos(8*pi*procufaxis));%,1,inputp.NT);
         fid = fid .* apdfunc; 
    end
end
end

